/*
 * Copyright (C) 2011 HBM Netherlands B.V.
 * Schutweg 15a
 * 5145NP Waalwijk
 * The Netherlands
 * http://www.hbm.com
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/*
 * HBM Acquisition Board driver.
 *
 * Maps PCI shared memory of Acquisition Board and makes this available to user space.
 */

// enable printing debugging info
//#define DEBUG

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/pci.h>
#include <linux/init.h>
#include <linux/ioport.h>
#include <asm/io.h>
#include <linux/interrupt.h>
#include <linux/slab.h>
#include <linux/cdev.h>
#include <linux/poll.h>
#include <linux/bug.h>
#include <linux/mm.h>

#include "hbm-acqboard-driver.h"
#include "hbm-acqboard.h"

#define MODNAME			"hbm-acqboard"
#define DRIVER_VERSION	"1.2"

// Vendor ID's are taken from <include/linux/pci_ids.h>!

// Device ID's
#define PCI_DEVICE_ID_IOP			(0x530d)		// 80310 IO Processor
#define PCI_DEVICE_ID_HBM			(0x1016)		// ?

// Sub vendor ID's
#define PCI_SUBVENDOR_ID_HBM		(0x11EF)		// Nicolet Technologies BV
#define PCI_SUBDEVICE_ID_ACQ		(0x5141)		// "AQ" Acquisition module

/* 
 *	CHARACTER DEVICES SECTION
 */
#define ACQBOARD_CHARDEV_MINOR_START 0
#define ACQBOARD_CHARDEV_MINOR_MAX	 32			// Max. 32 number of supported devices

static struct class * hbm_acqboard_class;
static dev_t hbm_acqboard_dev_t;
static int hbm_acqboard_minor_start = ACQBOARD_CHARDEV_MINOR_START;

//static int hbm_acqboard_chrdev_ioctl(struct inode *i, struct file *f, unsigned int cmd, unsigned long arg)
static long hbm_acqboard_chrdev_ioctl( struct file *f, unsigned int cmd, unsigned long arg)
{
	int ret = -EINVAL;
	struct inode *i = f->f_dentry->d_inode ;

	struct hbm_acqboard_dev *acqboard = (struct hbm_acqboard_dev*) container_of(i->i_cdev, struct hbm_acqboard_dev, cdev);
	BUG_ON(!acqboard);

	dev_dbg(&acqboard->pci_dev->dev, "hbm_acqboard_chrdev_ioctl");

	switch (cmd)
	{
		case HBM_ACQBOARD_IOCTL_READ_PHYSICAL_ADDRESS:

			dev_dbg(&acqboard->pci_dev->dev, "hbm_acqboard_chrdev_ioctl: HBM_ACQBOARD_IOCTL_READ_PHYSICAL_ADDRESS\n");

			if (copy_to_user((void *)arg, (void *)&(acqboard->acqboard_mem_base_phys), sizeof(int))!= 0)
			{
				// copy failed
				ret = -EFAULT;
			}
			else
			{
				ret = 0;
			}

			break;

		case HBM_ACQBOARD_IOCTL_READ_SUBSYSTEM_DEVICE_ID:

			dev_dbg(&acqboard->pci_dev->dev, "hbm_acqboard_chrdev_ioctl: HBM_ACQBOARD_IOCTL_READ_SUBSYSTEM_DEVICE_ID\n");

			if (copy_to_user((void *)arg, (void *)&(acqboard->subSystemDeviceId), sizeof(unsigned int))!= 0)
			{
				// copy failed
				ret = -EFAULT;
			}
			else
			{
				ret = 0;
			}

			break;

		case HBM_ACQBOARD_IOCTL_READ_MEMORY_SIZE:


			dev_dbg(&acqboard->pci_dev->dev, "hbm_acqboard_chrdev_ioctl %d: HBM_ACQBOARD_IOCTL_READ_MEMORY_SIZE\n",
						acqboard->acqboard_mem_size ) ;

			if (copy_to_user((void *)arg, (void *)&(acqboard->acqboard_mem_size), sizeof(unsigned int))!= 0)
			{
				// copy failed
				ret = -EFAULT;
			}
			else
			{
				ret = 0;
			}
			break ;


		default:
			// return: "inappropriate ioctl for device"
			ret = -ENOTTY;
			break;
	}

	return ret;
}

static int hbm_acqboard_chrdev_mmap(struct file *f, struct vm_area_struct *vma)
{
	struct hbm_acqboard_dev * acqboard;
	unsigned long phys_addr;
	unsigned long phys_size;

	BUG_ON(!f);
	BUG_ON(!vma);

	acqboard = (struct hbm_acqboard_dev*) f->private_data;
	BUG_ON(!acqboard);

	dev_dbg( &acqboard->pci_dev->dev, "hbm_acqboard_chrdev_mmap\n");

	// retrieve physical address of acquisition board memory (BAR0)
	phys_addr = acqboard->acqboard_mem_base_phys;
	phys_size = acqboard->acqboard_mem_size;

	dev_dbg(&acqboard->pci_dev->dev, "hbm_acqboard_chrdev_mmap: phys_mem_base: 0x%x, size: 0x%x", (unsigned int)phys_addr, (unsigned int)phys_size);

	if (remap_pfn_range(vma, vma->vm_start, phys_addr >> PAGE_SHIFT, phys_size, vma->vm_page_prot))
	{
		dev_err( &acqboard->pci_dev->dev, "hbm_acqboard_chrdev_mmap: failed");
		return -EAGAIN;
	}

	return 0;
}

static int hbm_acqboard_chrdev_open(struct inode *i, struct file *f)
{
	int ret = 0;
	struct hbm_acqboard_dev *acqboard = container_of(i->i_cdev, struct hbm_acqboard_dev, cdev);
	BUG_ON(!acqboard);

	dev_dbg( &acqboard->pci_dev->dev, "hbm_acqboard_chrdev_open minor: %d", acqboard->minor);

	// Check minor number
	if (iminor(i) != acqboard->minor)
	{
		/* Invalid minor, reject open() */
		ret = -EINVAL;
	}

	if (!ret)
	{
		f->private_data = acqboard;
	}
	
	return ret;
}

static int hbm_acqboard_chrdev_release(struct inode *i, struct file *f)
{
	struct hbm_acqboard_dev *acqboard = container_of(i->i_cdev, struct hbm_acqboard_dev, cdev);
	BUG_ON(!acqboard);

	dev_dbg( &acqboard->pci_dev->dev, "hbm_acqboard_chrdev_release");

	return 0;
}

struct file_operations hbm_acqboard_chrdev_fops = {
	.read		= NULL,				// not supported
	.write		= NULL,				// not supported
	.poll		= NULL,				// not supported
	.unlocked_ioctl 		= hbm_acqboard_chrdev_ioctl,
	.mmap		= hbm_acqboard_chrdev_mmap,
	.open		= hbm_acqboard_chrdev_open,
	.release	= hbm_acqboard_chrdev_release,
	.owner		= THIS_MODULE,
};

static int setup_character_device(struct hbm_acqboard_dev *acqboard)
{
	int ret = -EINVAL;

	BUG_ON(acqboard == NULL);
	BUG_ON(acqboard->pci_dev == NULL);

	dev_dbg(&acqboard->pci_dev->dev, "setup_character_device");

	// Check if max. number of supported devices is reached.
	if (hbm_acqboard_minor_start >= ACQBOARD_CHARDEV_MINOR_MAX)
	{
		dev_err(&acqboard->pci_dev->dev, "setup_character_device: max. number (%d) of supported devices reached!", hbm_acqboard_minor_start);
		return ret;
	}

	// Allocate the next free minor number form the pre-allocated region.
	acqboard->chrdev_region = MKDEV(MAJOR(hbm_acqboard_dev_t), hbm_acqboard_minor_start);

	dev_dbg(&acqboard->pci_dev->dev, "setup_character_device: chrdev_region: MAJOR: %d\n", MAJOR(acqboard->chrdev_region));
	dev_dbg(&acqboard->pci_dev->dev, "setup_character_device: chrdev_region: MINOR: %d\n", MINOR(acqboard->chrdev_region));

	cdev_init(&acqboard->cdev, &hbm_acqboard_chrdev_fops);

	acqboard->minor = MINOR(acqboard->chrdev_region);
	acqboard->cdev.owner = THIS_MODULE;

	// Add only 1 minor number to this device
	ret = cdev_add(&acqboard->cdev, acqboard->chrdev_region, 1);
	if (ret)
	{
		dev_err(&acqboard->pci_dev->dev, "setup_character_device: cdev_add failed (minor: %d): ret: %d\n", acqboard->minor, ret);

		kobject_put(&acqboard->cdev.kobj);

		return ret;
	}

	// Create device based on the module name and the minor number
	ret = (int)device_create(hbm_acqboard_class, &acqboard->pci_dev->dev, acqboard->chrdev_region, NULL, "%s%u", MODNAME, MINOR(acqboard->chrdev_region));
	if (ret == 0)
	{
		dev_err(&acqboard->pci_dev->dev, "setup_character_device: device_create failed (minor: %d): ret: %d\n", acqboard->minor, ret);

		cdev_del(&acqboard->cdev);

		kobject_put(&acqboard->cdev.kobj);

		return ret;
	}

	// Increment minor number for any further devices
	hbm_acqboard_minor_start++;

	return 0;
}


static int remove_character_devices(struct hbm_acqboard_dev *acqboard)
{
	BUG_ON(!acqboard);

	device_destroy(hbm_acqboard_class, acqboard->chrdev_region);

	cdev_del(&acqboard->cdev);

	return 0;
}


/*
	PCI DRIVER RELATED SECTION
*/

static struct pci_device_id hbm_acqboard_pci_tbl[] =
{
	// AcqMod1M, DigMod1M, AcqFib100M, AcqMod100M
	{ 	.vendor =  PCI_VENDOR_ID_INTEL,
		.device = PCI_DEVICE_ID_IOP,
		.subvendor = PCI_SUBVENDOR_ID_HBM,
		.subdevice = PCI_SUBDEVICE_ID_ACQ,
	},
	// AcqM250k
	{ 	.vendor =  PCI_VENDOR_ID_XILINX,
		.device = PCI_DEVICE_ID_HBM,
		.subvendor = PCI_SUBVENDOR_ID_HBM,
		.subdevice = PCI_SUBDEVICE_ID_ACQ,
	},
	{0,}
};
MODULE_DEVICE_TABLE (pci, hbm_acqboard_pci_tbl);

/**
 * THE pci device interrupt service routine handler.
 * Make sure that this function returns as fast as possible! Please use a 
 * work queue or a tasklet to execute more time consuming work...
*/
static irqreturn_t hbm_acqboard_irq_handler(int irq, void *dev_id)
{
	struct hbm_acqboard_dev *acqboard = (struct hbm_acqboard_dev *) dev_id;

	// No interrupts expected or supported!
	return IRQ_NONE;
}	

static int hbm_acqboard_pci_setup(struct pci_dev *pdev, struct hbm_acqboard_dev *acqboard)
{
	int ret = EINVAL;
	unsigned long tmp;
	unsigned short subSys;
	int err;

	BUG_ON(!pdev);
	BUG_ON(!acqboard);

	// Get BAR0 & determine BAR0 size
	acqboard->acqboard_mem_base_phys = pci_resource_start(pdev, 0);
	tmp = pci_resource_end(pdev, 0);	/* address is of last valid address */
	acqboard->acqboard_mem_size = tmp - acqboard->acqboard_mem_base_phys + 1; 

	dev_info(&pdev->dev, "Acqboard mem_base_phys: 0x%x, size: 0x%x", (unsigned int)acqboard->acqboard_mem_base_phys, (unsigned int)acqboard->acqboard_mem_size);
	//TODO: check mem_size validity!!

	// Get flags
	tmp = pci_resource_flags(pdev, 0);
	BUG_ON(0 == (tmp & IORESOURCE_MEM)); /* This chunk must be MEM */

	acqboard->acqboard_mem_base_virt = ioremap_nocache(acqboard->acqboard_mem_base_phys, acqboard->acqboard_mem_size);
	if (!acqboard->acqboard_mem_base_virt)
	{
		dev_err(&pdev->dev, "Could not ioremap() acqboard PCI memory area (phys = 0x%08lx, "
				"size=0x%04x)", acqboard->acqboard_mem_base_phys, acqboard->acqboard_mem_size);

		goto pci_setup_error;
	}

	subSys = 0;
	err = pci_read_config_word(pdev, 0x2e, &subSys);
	if (err != 0)
	{
		dev_err(&pdev->dev, "Acqboard: pci_read_config_word failed: err: 0x%d\n", err);
		goto pci_setup_error;
	}

	dev_info(&pdev->dev, "Acqboard sub device ID: 0x%x\n", (unsigned int)subSys);
	acqboard->subSystemDeviceId = (unsigned int)subSys;

	return 0;

pci_setup_error:
	if(acqboard->acqboard_mem_base_virt)
		iounmap(acqboard->acqboard_mem_base_virt);
	if(acqboard->irq)
		free_irq(acqboard->irq, acqboard);
	return ret;
}


static int __devinit hbm_acqboard_pci_probe(struct pci_dev *pdev,
				       const struct pci_device_id *ent)
{
	struct hbm_acqboard_dev *acqboard = NULL;
	int ret;

	dev_info(&pdev->dev, "Found HBM Acquisition Board %d", hbm_acqboard_minor_start);

	/* Now allocate (zeroed) memory for our internal bookkeeping */
	acqboard = kzalloc( sizeof(struct hbm_acqboard_dev), GFP_KERNEL);
	if (!acqboard)
	{
		dev_err( &pdev->dev, "Could not allocate struct hbm_acqboard_dev, aborting...");
		ret = -ENOMEM;
		goto probe_out;
	}

	pci_set_drvdata (pdev, acqboard);
	pci_enable_device(pdev); // TODO: check return value;

	ret = pci_request_regions(pdev, MODNAME);
	if (ret)
	{
		dev_err( &pdev->dev, "Could not claim pci device related regions, aborting... (err = %d / 0x%02x)", ret, ret);
		goto probe_out_disable;
	}

	acqboard->pci_dev = pdev;	

	ret = hbm_acqboard_pci_setup(pdev, acqboard);
	if (ret)
	{
		dev_err( &pdev->dev, "hbm_acqboard_pci_setup() failed, aborting... (err = %d / 0x%02x)", ret, ret);
		goto probe_out_release;
	}

	// Enable bus-mastering
	pci_set_master(pdev);

	// Setup character device
	ret = setup_character_device(acqboard);
	if (ret)
	{
		dev_err( &pdev->dev, "Could not setup character devices, aborting... (err = %d / 0x%02x)", ret, ret);
		goto probe_out_free;
	}

	dev_dbg( &pdev->dev, "hbm_acqboard_pci_probe() finished with success.");

	return 0;

probe_out_chardevs:
	remove_character_devices(acqboard);
probe_out_release:
	pci_release_regions(pdev);
probe_out_disable:
	pci_disable_device(pdev);
probe_out_free:
	if (acqboard)
	{
		pci_set_drvdata(pdev, NULL);
		kfree(acqboard);
	}
probe_out:
	dev_err( &pdev->dev, "hbm_acqboard_pci_probe() failed... (err = %d / 0x%02x)", ret, ret);
	return ret;
}

static void __devexit hbm_acqboard_pci_remove(struct pci_dev *pdev)
{
	struct hbm_acqboard_dev *acqboard =  pci_get_drvdata(pdev);

	pci_set_drvdata(pdev, NULL);

	remove_character_devices(acqboard);

	pci_clear_master(pdev);
	pci_release_regions(pdev);
	pci_disable_device(pdev);

	if (acqboard->acqboard_mem_base_virt)
		iounmap(acqboard->acqboard_mem_base_virt);
	if (acqboard->irq)
		free_irq(acqboard->irq, acqboard);

	kfree(acqboard);
}


#ifdef CONFIG_PM
/* 	The following functions are for power management ( PM )	*/

static int hbm_acqboard_pci_suspend(struct pci_dev *pdev, pm_message_t state)
{
	struct hbm_acqboard_dev *dev =  pci_get_drvdata(pdev);

	//TODO: power down / suspend the acq board hardware

	/* TODO: found out what these next 2 lines do */
	pci_save_state(pdev);
	pci_set_power_state(pdev, PCI_D3hot);

	return 0;	/* Never return anything else since it will prevent */	
			/* the kernel from going into suspend mode! */
}

static int hbm_acqboard_pci_resume(struct pci_dev *pdev)
{
	struct hbm_acqboard_dev *dev =  pci_get_drvdata(pdev);

	pci_set_power_state(pdev, PCI_D0);
	pci_restore_state(pdev);
	return 0;
}
#endif /* CONFIG_PM */

static struct pci_driver hbm_acqboard_pci_driver = {
	.name		= MODNAME,
	.id_table	= hbm_acqboard_pci_tbl,
	.probe		= hbm_acqboard_pci_probe,
	.remove		= __devexit_p(hbm_acqboard_pci_remove),
#ifdef CONFIG_PM
//	.suspend	= hbm_acqboard_pci_suspend,
//	.resume		= hbm_acqboard_pci_resume,
#endif /* CONFIG_PM */
};

static int __init hbm_acqboard_init_module(void)
{
	int rv = 0;

	printk("HBM Acquisition Board Driver: version: %s, Build date: %s %s\n", DRIVER_VERSION, __TIME__, __DATE__);

	// Pre-allocate major and minor range (0 - 31)
	rv = alloc_chrdev_region(&hbm_acqboard_dev_t, ACQBOARD_CHARDEV_MINOR_START, ACQBOARD_CHARDEV_MINOR_MAX, MODNAME);
	if (rv)
	{
		/* Could not allocate major/minor region for this device */
		printk("Could not allocate major/minor region for this device");
		return rv;
	}

	hbm_acqboard_class = class_create(THIS_MODULE, MODNAME);

	rv = pci_register_driver(&hbm_acqboard_pci_driver);

	return rv;
}

static void __exit hbm_acqboard_cleanup_module(void)
{
	unregister_chrdev_region(hbm_acqboard_dev_t, ACQBOARD_CHARDEV_MINOR_MAX);

	class_destroy(hbm_acqboard_class);

	pci_unregister_driver(&hbm_acqboard_pci_driver);
}

MODULE_AUTHOR("Jeroen van den Berg <jbe@chess.nl>");
MODULE_DESCRIPTION("HBM Acquisition Board driver");
MODULE_LICENSE("GPL");

module_init(hbm_acqboard_init_module);
module_exit(hbm_acqboard_cleanup_module);
