/*
 *	w83627HF based driver to sample the Boot Mode Switch on the backplane.
 *	Added ioctl for general GPIO access. (jbe)
 *
 * Copyright (C) 2011 HBM Netherlands B.V.
 * Schutweg 15a
 * 5145NP Waalwijk
 * The Netherlands
 * http://www.hbm.com
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 *	Based on: 
 *	(c) Copyright 2007 Vlad Drukker <vlad@storewiz.com>
 *		added support for W83627THF.
 *
 *	(c) Copyright 2003,2007 P�draig Brady <P@draigBrady.com>
 *
 *	Based on advantechwdt.c which is based on wdt.c.
 *	Original copyright messages:
 *
 *	(c) Copyright 2000-2001 Marek Michalkiewicz <marekm@linux.org.pl>
 *
 *	(c) Copyright 1996 Alan Cox <alan@lxorguk.ukuu.org.uk>,
 *						All Rights Reserved.
 *
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public License
 *	as published by the Free Software Foundation; either version
 *	2 of the License, or (at your option) any later version.
 *
 *	Neither Alan Cox nor CymruNet Ltd. admit liability nor provide
 *	warranty for any of this software. This material is provided
 *	"AS-IS" and at no charge.
 *
 *	(c) Copyright 1995    Alan Cox <alan@lxorguk.ukuu.org.uk>
 */

#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/types.h>
#include <linux/device.h>
#include <linux/miscdevice.h>
#include <linux/watchdog.h>
#include <linux/fs.h>
#include <linux/ioport.h>
#include <linux/notifier.h>
#include <linux/reboot.h>
#include <linux/init.h>
#include <linux/spinlock.h>
#include <linux/io.h>
#include <linux/uaccess.h>
#include "hbm-bootmode-driver.h"

//#include <asm/system.h>

#define MODULE_NAME HBM_BOOTMODE_DEVICE_NAME
#define PFX MODULE_NAME ": "

#define DRIVER_VERSION	"1.3"

/* You must set this - there is no sane way to probe for this board. */
/* The actual address is determined by the configuration on reset of the SuperIO chip */
#define DEV_IO_DEFAULT_VALUE	0x2e
#define DEV_IO_ALT_VALUE		0x4e
static int dev_io = DEV_IO_DEFAULT_VALUE;
module_param(dev_io, int, 0);
MODULE_PARM_DESC(dev_io, "w83627hf/thf GPIO io port (default " "DEV_IO_DEFAULT_VALUE" ")");

#define GPIO1_LOGICAL_DEVNR	(7)
#define GPIO2_LOGICAL_DEVNR	(8)
#define GPIO3_LOGICAL_DEVNR	(9)

#define GPIO34_MASK		(1<<4)

static volatile long unsigned int module_is_open=0;
static int selected_port = -1;

/*
 *	Kernel methods.
 */

#define EFER_OFFSET	0
#define EFIR_OFFSET	EFER_OFFSET
#define EFDR_OFFSET	1

#define EFER (dev_io+EFER_OFFSET)	/* Extended Function Enable Registers */
#define EFIR (dev_io+EFIR_OFFSET)	/* Extended Function Index Register (same as EFER) */
#define EFDR (dev_io+EFDR_OFFSET)	/* Extended Function Data Register */



static void superio_enter(unsigned int base_addr)
{
         outb(0x87, base_addr);
         outb(0x87, base_addr);
}

static void w83627hf_select_register(unsigned char logic_dev_nr)
{
	outb_p(0x87, EFER); /* Enter extended function mode */
	outb_p(0x87, EFER); /* Again according to manual */


	outb_p(0x07, EFER); /* point to logical device number reg */
	outb_p(logic_dev_nr, EFDR); /* select logical device */
}

static void w83627hf_unselect_register(void)
{
	outb_p(0xAA, EFER); /* Leave extended function mode */
}

static unsigned char read_bootmode_status(void)
{
	unsigned char c;
	w83627hf_select_register(GPIO3_LOGICAL_DEVNR);
	
	/* Read register CRF1, bit 4 contains input value of GP34 */
	outb_p( 0xF1, EFER );
	c = inb(EFDR);
	//printk( PFX "read_bootmode_status(): 0x%02x\n", c);
	w83627hf_unselect_register();
	return (c & GPIO34_MASK ? 1 : 0 );
}

static void write_bootmode_status(unsigned char c)
{
	unsigned char tmp;
	w83627hf_select_register(GPIO3_LOGICAL_DEVNR);
	
	/* Read register CRF1, bit 4 contains input value of GP34 */
	outb_p( 0xF1, EFER );
	tmp = inb(EFDR);
	if(c)
		tmp |= GPIO34_MASK;
	else
		tmp &= ~GPIO34_MASK;
	outb_p( 0xF1, EFER );
	outb_p( tmp, EFDR );
	//printk( PFX "write_bootmode_status(): 0x%02x\n", tmp);
	w83627hf_unselect_register();
}

static unsigned char read_bootmode_config(void)
{
	unsigned char c;
	w83627hf_select_register(GPIO3_LOGICAL_DEVNR);
	
	/* Read register CRF0, bit 4 contains input/output setting of GP34 */
	outb_p( 0xF0, EFER );
	c = inb(EFDR);
	//printk( PFX "read_bootmode_config(): 0x%02x\n", c);
	w83627hf_unselect_register();
	return (c & GPIO34_MASK ? 1 : 0 );
}

static void write_bootmode_config(unsigned char c)
{
	unsigned char tmp;
	w83627hf_select_register(GPIO3_LOGICAL_DEVNR);
	
	/* Read register CRF0, bit 4 contains input/output setting of GP34 */
	outb_p( 0xF0, EFER );
	tmp = inb(EFDR);
	//printk( PFX "write_bootmode_config(): 0x%02x\n", c);
	if(c)
		tmp |= GPIO34_MASK;
	else
		tmp &= ~GPIO34_MASK;
	
	outb_p( 0xF0, EFER );
	outb_p( tmp, EFDR );
	w83627hf_unselect_register();
}

static unsigned int bootmode_getGPIO(unsigned int gpio)
{
	unsigned char c;
	unsigned char bitPos;

	switch (gpio)
	{
		case 10:
		case 11:
		case 12:
		case 13:
		case 14:
		case 15:
		case 16:
		case 17:
			bitPos = (gpio - 10);

			/* Enable Logical device GPIO1_LOGICAL_DEVNR */
			w83627hf_select_register(GPIO1_LOGICAL_DEVNR);

			break;

		case 20:
		case 21:
		case 22:
		case 23:
		case 24:
		case 25:
		case 26:
		case 27:
			bitPos = (gpio - 20);

			/* Enable Logical device GPIO2_LOGICAL_DEVNR */
			w83627hf_select_register(GPIO2_LOGICAL_DEVNR);

			break;

		case 30:
		case 31:
		case 32:
		case 33:
		case 34:
		case 35:
			bitPos = (gpio - 30);

			/* Enable Logical device GPIO3_LOGICAL_DEVNR */
			w83627hf_select_register(GPIO3_LOGICAL_DEVNR);

			break;

		default:
			// error, unsupported gpio
			return 0;
			break;
	}

	outb_p(0xF1, EFER); /* select CRF1 */
	c = inb(EFDR);	/* get data */

	w83627hf_unselect_register();

	// data in bit 0
	c = (c >> bitPos) & 0x1;

	return c;
}

// value: bit 0 containes data
static void bootmode_setGPIO(unsigned int gpio, unsigned int value)
{
	unsigned char c;
	unsigned char bitPos;

	switch (gpio)
	{
		case 10:
		case 11:
		case 12:
		case 13:
		case 14:
		case 15:
		case 16:
		case 17:
			bitPos = (gpio - 10);

			/* Enable Logical device GPIO1_LOGICAL_DEVNR */
			w83627hf_select_register(GPIO1_LOGICAL_DEVNR);

			break;

		case 20:
		case 21:
		case 22:
		case 23:
		case 24:
		case 25:
		case 26:
		case 27:
			bitPos = (gpio - 20);

			/* Enable Logical device GPIO2_LOGICAL_DEVNR */
			w83627hf_select_register(GPIO2_LOGICAL_DEVNR);

			break;

		case 30:
		case 31:
		case 32:
		case 33:
		case 34:
		case 35:
			bitPos = (gpio - 30);

			/* Enable Logical device GPIO3_LOGICAL_DEVNR */
			w83627hf_select_register(GPIO3_LOGICAL_DEVNR);

			break;

		default:
			// error, unsupported gpio
			return;
			break;
	}

	outb_p(0xF1, EFER); /* select CRF1 */
	c = inb(EFDR);	/* get data */

	if ((value & 0x1) == 0)
	{
		// clear bit
		c &= ~(1 << bitPos);
	}
	else
	{
		// set bit
		c |= (1 << bitPos);
	}

	outb_p( 0xF1, EFER );
	outb_p( c, EFDR );

	w83627hf_unselect_register();
}

static void w83627hf_init(void)
{
	unsigned char c;

	// GPIO1

	/* Enable Logical device GPIO1_LOGICAL_DEVNR */
	w83627hf_select_register(GPIO1_LOGICAL_DEVNR);
	outb_p(0x30, EFER); /* select CR30 */
	outb_p(0x01, EFDR); /* set bit 0 to activate this logical device */

	// initial values: GPIO10,16: 0 GPIO17: 1!
	c = 0x80;

	/* Write initial data to outputs, register CRF1*/
	outb_p(0xF1, EFER );
	outb_p( c, EFDR );


	/* Disable inversion */
	c = 0x0;
	outb_p(0xF2, EFER );
	outb_p(c, EFDR );

	// Set GPIO10 .. GPIO17 to output ('0')
	c = 0x00;

	/* write register CRF0 */
	outb_p( 0xF0, EFER );
	outb_p( c, EFDR );

	/* Last action (to prevent glitches), set mux */

	/* MUX Enable bit 10..17 van CR 2A	*/
	outb_p(0x2A, EFER);
	c = inb(EFDR);
	c |= (0xFD);			// save bit 1
	outb_p(0x2A, EFER );
	outb_p( c, EFDR );

	w83627hf_unselect_register();


	// GPIO2

	/* Enable Logical device GPIO2_LOGICAL_DEVNR */
	w83627hf_select_register(GPIO2_LOGICAL_DEVNR);
	outb_p(0x30, EFER); /* select CR30 */
	outb_p(0x01, EFDR); /* set bit 0 to activate this logical device */

	// initial values: GPIO22,25,26: 0 GPIO27: 0!
	c = 0;

	/* Write initial data to outputs, register CRF1*/
	outb_p(0xF1, EFER );
	outb_p( c, EFDR );

	/* Disable inversion */
	c = 0x0;
	outb_p(0xF2, EFER );
	outb_p(c, EFDR );

	/* Read register CRF0*/
	outb_p( 0xF0, EFER );
	c = inb(EFDR);

	// Set GPIO22, GPIO25, GPIO26 and GPIO27 to output ('0'), rest to input
	c = 0x1B;

	/* write register CRF0 */
	outb_p( 0xF0, EFER );
	outb_p( c, EFDR );

	/* Last action (to prevent glitches), set mux */

	/* MUX Enable bit 0..7 van CR 2B	*/
	c = 0xFF;
	outb_p(0x2B, EFER );
	outb_p( c, EFDR );

	w83627hf_unselect_register();


	// GPIO3

	/* Enable Logical device GPIO3_LOGICAL_DEVNR */
	w83627hf_select_register(GPIO3_LOGICAL_DEVNR);
	outb_p(0x30, EFER); /* select CR30 */
	outb_p(0x01, EFDR); /* set bit 0 to activate this logical device */

	/* Disable inversion */
	outb_p(0xF2, EFER );
	c = inb(EFDR);
	c &= ~0xFC; /* clear bit 2-7 to disable inversion of GPIO30-35 */
	outb_p(0xF2, EFER );
	outb_p(c, EFDR );

	// CRF1, GPIO port 3, bit 7-6 reserved, data on 5:0
	// initial values: GPIO30, 32, 35: 1
	c = (1 << 5) | (1 << 2) | (1 << 0);

	/* Write initial data to outputs, register CRF1*/
	outb_p(0xF1, EFER );
	outb_p( c, EFDR );

	/* Read register CRF0*/
	outb_p( 0xF0, EFER );
	c = inb(EFDR);

	// Set GPIO30, GPIO32, GPIO35 to output ('0'), rest input
	c = 0xDA;

	/* write register CRF0 */
	outb_p( 0xF0, EFER );
	outb_p( c, EFDR );

	/* Last action (to prevent glitches), set mux */

	/* MUX Enable bit 2..7 van CR 29	*/
	outb_p(0x29, EFER);
	c = inb(EFDR);
	c |= (0xFC);
	outb_p(0x29, EFER );
	outb_p( c, EFDR );

	w83627hf_unselect_register();
}


static int bootmode_open(struct inode *inode, struct file *file)
{
	if (test_and_set_bit(0, &module_is_open))
		return -EBUSY;
	/*
	 *	Activate
	 */

	return nonseekable_open(inode, file);
}

static int bootmode_close(struct inode *inode, struct file *file)
{
	clear_bit(0, &module_is_open);
	return 0;
}

static int bootmode_ioctl(struct file *f, unsigned int cmd, unsigned long arg)
{
	int ret = -EINVAL;
	struct gpioData data;
	unsigned char c;
	unsigned char bitPos;
	
	switch (cmd)
	{
		// Read specified bit
		case HBM_BOOTMODE_IOCTL_READ_GPIO:

//			printk("bootmode_ioctl: BOOTMODE_READ_GPIO\n");

			// copy data from user
			if (copy_from_user((void *)(&data), (void *)arg, sizeof(data))!= 0)
			{
				// copy failed
				ret = -EFAULT;

				break;
			}
			else
			{
				ret = 0;
			}

			// check GPIO number
			if (((data.gpioNumber >= 10) && (data.gpioNumber <= 17)) ||
				((data.gpioNumber >= 20) && (data.gpioNumber <= 27)) ||
				((data.gpioNumber >= 30) && (data.gpioNumber <= 35)))
			{
				// get gpio value
				data.value = bootmode_getGPIO(data.gpioNumber);

				// transfer data to user
				if (copy_to_user((void *)arg, (void *)(&data), sizeof(data))!= 0)
				{
					// copy failed
					ret = -EFAULT;
				}
				else
				{
					ret = 0;
				}
			}
			else
			{
				// invalid parameter
				ret = -EINVAL;
			}

			break;

		// write specified bit
		case HBM_BOOTMODE_IOCTL_WRITE_GPIO:

//			printk("bootmode_ioctl: BOOTMODE_WRITE_GPIO\n");

			// copy data from user
			if (copy_from_user((void *)(&data), (void *)arg, sizeof(data))!= 0)
			{
				printk("bootmode_ioctl: cinvalid parameter\n");

				// copy failed
				ret = -EFAULT;
			}
			else
			{
				ret = 0;
			}

			//TODO: check input/output?

			// check GPIO number
			if (((data.gpioNumber >= 10) && (data.gpioNumber <= 17)) ||
				((data.gpioNumber >= 20) && (data.gpioNumber <= 27)) ||
				((data.gpioNumber >= 30) && (data.gpioNumber <= 35)))
			{
				// get gpio value
				bootmode_setGPIO(data.gpioNumber, data.value);

				ret = 0;
			}
			else
			{
				// invalid parameter
				ret = -EINVAL;
			}

			break;

		case HBM_BOOTMODE_IOCTL_TOGGLE_GPIO:
//			printk("bootmode_ioctl: BOOTMODE_TOGGLE_GPIO\n");
			
			// copy data from user
			if (copy_from_user((void *)(&data), (void *)arg, sizeof(data))!= 0)
			{
				printk("bootmode_ioctl: cinvalid parameter\n");

				// copy failed
				ret = -EFAULT;
			}
			else
			{
				ret = 0;
			}

			switch (data.gpioNumber)
			{
				case 10:
				case 11:
				case 12:
				case 13:
				case 14:
				case 15:
				case 16:
				case 17:
					bitPos = (data.gpioNumber - 10);

					/* Enable Logical device GPIO1_LOGICAL_DEVNR */
					w83627hf_select_register(GPIO1_LOGICAL_DEVNR);

					break;

				case 20:
				case 21:
				case 22:
				case 23:
				case 24:
				case 25:
				case 26:
				case 27:
					bitPos = (data.gpioNumber - 20);

					/* Enable Logical device GPIO2_LOGICAL_DEVNR */
					w83627hf_select_register(GPIO2_LOGICAL_DEVNR);

				break;

				case 30:
				case 31:
				case 32:
				case 33:
				case 34:
				case 35:
					bitPos = (data.gpioNumber - 30);
	
					/* Enable Logical device GPIO3_LOGICAL_DEVNR */
					w83627hf_select_register(GPIO3_LOGICAL_DEVNR);

					break;

				default:
					// error, unsupported gpio
					return;
					break;
			}

			outb_p(0xF1, EFER); 	/* select CRF1 (data register) */
			c = inb(EFDR);		/* get data */

			{
				int i;

				for (i=0; i<1000; i++)
				{
					// clear bit
					c &= ~(1 << bitPos);
					outb_p( c, EFDR );

					// set bit
					c |= (1 << bitPos);
					outb_p( c, EFDR );
				}
				ret = 0;
			}

			w83627hf_unselect_register();

			break;

		case HBM_BOOTMODE_IOCTL_SELECT_PORT:
			// copy data from user
			if (copy_from_user((void *)(&data), (void *)arg, sizeof(data))!= 0)
			{
				printk("bootmode_ioctl: cinvalid parameter\n");

				// copy failed
				ret = -EFAULT;
			}
			else
			{
				ret = 0;
			}

			switch (data.gpioPort)
			{
				case 1:
					/* Enable Logical device GPIO1_LOGICAL_DEVNR */
					w83627hf_select_register(GPIO1_LOGICAL_DEVNR);

					break;

				case 2:
					/* Enable Logical device GPIO2_LOGICAL_DEVNR */
					w83627hf_select_register(GPIO2_LOGICAL_DEVNR);

					break;

				case 3:
					/* Enable Logical device GPIO3_LOGICAL_DEVNR */
					w83627hf_select_register(GPIO3_LOGICAL_DEVNR);

					break;

				default:
					// error, unsupported gpio port
					ret = -EINVAL;
					return ret;
					break;
			}

			// store selected port in static variable 
			selected_port = data.gpioPort;

			outb_p(0xF1, EFER); 	/* select CRF1 (data register) */

			break;

		case HBM_BOOTMODE_IOCTL_WRITE_PORT:
			/* write data to previously (!) selected port */

			// copy data from user
			if (copy_from_user((void *)(&data), (void *)arg, sizeof(data))!= 0)
			{
				printk("bootmode_ioctl: cinvalid parameter\n");

				// copy failed
				ret = -EFAULT;
			}
			else
			{
				ret = 0;
			}
			
			if (selected_port == data.gpioPort)
			{
				c = (unsigned char)data.portValue;
				outb_p( c, EFDR );
			}
			else
			{
				ret = -EINVAL;
			}

			break;

		case HBM_BOOTMODE_IOCTL_READ_PORT:
			/* Read data from previously (!) selected port */
			// copy data from user
			if (copy_from_user((void *)(&data), (void *)arg, sizeof(data))!= 0)
			{
				printk("bootmode_ioctl: cinvalid parameter\n");

				// copy failed
				ret = -EFAULT;
			}
			else
			{
				ret = 0;
			}

			if (selected_port == data.gpioPort)
			{
				// get data
				c = inb(EFDR);
				data.portValue = (unsigned int)c;

				// transfer data to user
				if (copy_to_user((void *)arg, (void *)(&data), sizeof(data))!= 0)
				{
					// copy failed
					ret = -EFAULT;
				}
				else
				{
					ret = 0;
				}
			}
			else
			{
				ret = -EINVAL;
			}

			break;
		
		case HBM_BOOTMODE_IOCTL_DESELECT_PORT:
			// copy data from user
			if (copy_from_user((void *)(&data), (void *)arg, sizeof(data))!= 0)
			{
				printk("bootmode_ioctl: cinvalid parameter\n");

				// copy failed
				ret = -EFAULT;
			}
			else
			{
				ret = 0;
			}

			if (selected_port == data.gpioPort)
			{
				w83627hf_unselect_register();
				selected_port = -1;
				ret = 0;
			}
			else
			{
				ret = -EINVAL;
			}

			break;

		default:
			// return: "inappropriate ioctl for device"
			ret = -ENOTTY;
			break;
	}

	return ret;
}


/*
 *	Sysfs stuff
 */ 

static ssize_t bootmode_status_show( struct device *dev, struct device_attribute *attr, char *buf)
{
	unsigned char c;

	c = read_bootmode_status();
	return snprintf(buf, PAGE_SIZE, "%i", c);
}

#ifdef DEBUG_DRIVER
// We only need these functions if we want the debug stuff
// Furthermore we only need to write in debug mode
static ssize_t bootmode_status_store(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
	unsigned char c;

	if( 0 == strncmp("0", buf, count)){
		c = 0;
	} else {
		c = 1;
	}
	write_bootmode_status(c);
	return count;
}
static ssize_t bootmode_config_show( struct device *dev, struct device_attribute *attr, char *buf)
{
	unsigned char c;

	c = read_bootmode_config();
	return snprintf(buf, PAGE_SIZE, "%i", c);
}

static ssize_t bootmode_config_store(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{		
	unsigned char c;

	if( 0 == strncmp("0", buf, count)){
		c = 0;
	} else {
		c = 1;
	}
	write_bootmode_config(c);
	return count;
}
static DEVICE_ATTR(bootmode_stat, S_IRUGO | S_IWUGO, bootmode_status_show, bootmode_status_store);
static DEVICE_ATTR(bootmode_conf, S_IRUGO | S_IWUGO, bootmode_config_show, bootmode_config_store);
#else
static DEVICE_ATTR(bootmode_stat, S_IRUGO, bootmode_status_show, NULL );
#endif


/*
 *	Kernel Interfaces
 */

static const struct file_operations bootmode_fops = {
	.owner		= THIS_MODULE,
	.llseek		= no_llseek,
//	.write		= wdt_write,
//	.unlocked_ioctl	= dev_ioctl,
	.open		= bootmode_open,
	.release	= bootmode_close,
	.unlocked_ioctl = bootmode_ioctl
};

static struct miscdevice bootmode_miscdev = {
	.minor = MISC_DYNAMIC_MINOR,
	.name = MODULE_NAME,
	.fops = &bootmode_fops,
};


static int __init probe_at_addr(unsigned int base_addr){
	unsigned char hi, lo;

	superio_enter( base_addr ) ;
	outb_p(0x20, base_addr + EFER_OFFSET);
	hi = inb(base_addr + EFDR_OFFSET);
	outb_p(0x21, base_addr + EFER_OFFSET);
	lo = inb(base_addr + EFDR_OFFSET);
	printk( KERN_INFO PFX "Probing for CHIPID @ IO addr 0x%02x: 0x%04x (Rev %c) \n", base_addr, hi <<8 | lo, lo);

	return (hi==0x52? 0 : -ENXIO);
}

static int __init bootmode_init(void)
{
	int ret;
	printk(KERN_INFO "HBM IM2 Boot Mode driver using W83627HF/THF/HG Super I/O chip. Probing started...\n");
	printk("HBM IM2 Boot Mode driver: version: %s, Build date: %s %s\n", DRIVER_VERSION, __TIME__, __DATE__);
	if(probe_at_addr(DEV_IO_DEFAULT_VALUE)){
		ret = probe_at_addr(DEV_IO_ALT_VALUE);
		if(ret){
			printk(KERN_INFO "HBM IM2 Boot Mode driver: No Super I/O chip found. Exitting....\n");
			goto out;
		}
		dev_io = DEV_IO_ALT_VALUE;
	}
	else{
		dev_io = DEV_IO_DEFAULT_VALUE;	
	}
	printk(KERN_INFO PFX "Found chip @ 0x%02x\n", dev_io);

	if (!request_region(dev_io, 2, MODULE_NAME)) {
		printk(KERN_ERR PFX "I/O address 0x%04x already in use\n", dev_io);
		ret = -EIO;
		goto out;
	}

	printk(KERN_INFO PFX "Initializing...");
	w83627hf_init();

	ret = misc_register(&bootmode_miscdev);
	if (ret != 0) {
		printk(KERN_ERR PFX
			"cannot register miscdev on minor=%d (err=%d)\n",
							MISC_DYNAMIC_MINOR, ret);
		goto unreg_regions;
	}
	
	ret = device_create_file(bootmode_miscdev.this_device, &dev_attr_bootmode_stat);
	if(ret) {
		printk(KERN_ERR PFX "Could not create sysfs file: bootmode_stat");
		goto unreg_misc;
	}

#ifdef DEBUG_DRIVER
	ret = device_create_file(bootmode_miscdev.this_device, &dev_attr_bootmode_conf);
	if(ret) {
		printk(KERN_ERR PFX "Could not create sysfs file: bootmode_conf");
		device_remove_file(bootmode_miscdev.this_device, &dev_attr_bootmode_stat);
		goto unreg_misc;
	}
#endif
	printk(KERN_INFO PFX "Done. \n");

out:
	return ret;
unreg_misc:
	misc_deregister(&bootmode_miscdev);
unreg_regions:
	printk(KERN_INFO "HBM IM2 Boot Mode driver: release_region\n");
	release_region(dev_io, 2);
	goto out;
}

static void __exit bootmode_exit(void)
{
	
	device_remove_file(bootmode_miscdev.this_device, &dev_attr_bootmode_stat);
#ifdef DEBUG_DRIVER
	device_remove_file(bootmode_miscdev.this_device, &dev_attr_bootmode_conf);
#endif
	misc_deregister(&bootmode_miscdev);
	release_region(dev_io, 2);
}

module_init(bootmode_init);
module_exit(bootmode_exit);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Jeroen van den Berg <jbe@chess.nl>");
MODULE_DESCRIPTION("HBM Boot mode driver");
