/*
 * HBM IntfMod2 Ethernet core driver.
 *
 * Handles the Ehternet core registration and provide register mapping.
 *
 * Copyright (C) 2011 HBM Netherlands B.V.
 * Schutweg 15a
 * 5145NP Waalwijk
 * The Netherlands
 * http://www.hbm.com
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
// #define DEBUG

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/platform_device.h>
#include <linux/pci.h>
#include <linux/init.h>
#include <linux/ioport.h>
#include <asm/io.h>
#include <linux/interrupt.h>
#include <linux/slab.h>
#include <linux/cdev.h>
#include <linux/poll.h>
#include <linux/bug.h>
#include <linux/gfp.h>
#include <linux/kfifo.h>
#include <linux/wait.h>
#include <linux/sched.h>
#include <linux/wait.h>

#include "hbm-eth-core.h"

#define MODNAME			"hbm-eth-core"
#define DRIVER_VERSION	"1.1"

#define ETH_CHARDEV_MINOR_SIZE 1

/* 
	CHARACTER DEVICES SECTION
*/

#define SWITCH_MINOR(i)	switch(iminor(i))

static int hbm_eth_chrdev_mmap(struct file *f, struct vm_area_struct *vma)
{
	unsigned long phys_addr;
	unsigned long phys_size;
	struct hbm_eth_dev * eth = NULL;
	eth = (struct hbm_eth_dev *) f->private_data;

	BUG_ON(!f);
	BUG_ON(!vma);
	BUG_ON(!eth);

	phys_addr = eth->eth_mem_base_phys;
	phys_size = eth->eth_mem_size;

	if (remap_pfn_range(vma, vma->vm_start, phys_addr >> PAGE_SHIFT, phys_size, vma->vm_page_prot))
	{
		dev_info(&eth->pci_dev->dev, "hbm_eth_chrdev_mmap: failed\n");
		return -EAGAIN;
	}

	return 0;
}

static int hbm_eth_chrdev_open(struct inode *i, struct file *f)
{
	int ret = 0;
	struct hbm_eth_dev *eth = container_of(i->i_cdev, struct hbm_eth_dev, cdev);

	BUG_ON(!eth);
	dev_dbg(&eth->pci_dev->dev, "hbm_eth_chrdev_open\n");

	f->private_data = eth;

	dev_dbg(&eth->pci_dev->dev, "hbm_eth_chrdev_open: end\n");	

	return ret;
}

static int hbm_eth_chrdev_release(struct inode *i, struct file *f)
{
	int ret = 0;
	struct hbm_eth_dev *eth = container_of(i->i_cdev, struct hbm_eth_dev, cdev);

	BUG_ON(!eth);

	dev_dbg(&eth->pci_dev->dev, "hbm_eth_chrdev_release\n");

	return ret;
}

struct file_operations hbm_eth_chrdev_fops = {
	.read		= NULL,			// Not supported
	.write		= NULL,			// Not supported
	.poll		= NULL,			// Not supported
	.unlocked_ioctl		= NULL,			// Not supported
	.mmap		= hbm_eth_chrdev_mmap,
	.open		= hbm_eth_chrdev_open,
	.release	= hbm_eth_chrdev_release,
	.owner		= THIS_MODULE,
};

static int setup_character_devices(struct hbm_eth_dev *eth)
{
	int ret;

	BUG_ON(eth == NULL);
	BUG_ON(eth->pci_dev == NULL);

	ret = alloc_chrdev_region( &eth->chrdev_region, 0, ETH_CHARDEV_MINOR_SIZE, MODNAME);
	if (ret)
	{
		/* Could not allocate major/minor region for this device */
		dev_err( &eth->pci_dev->dev, "Could not allocate major/minor region for this device");
		return ret;
	}

	cdev_init(&eth->cdev, &hbm_eth_chrdev_fops);
	eth->cdev.owner = THIS_MODULE;

	ret = cdev_add(&eth->cdev, eth->chrdev_region, ETH_CHARDEV_MINOR_SIZE);
	if (ret)
	{
		kobject_put(&eth->cdev.kobj);
		unregister_chrdev_region(eth->chrdev_region, ETH_CHARDEV_MINOR_SIZE);        
		dev_err( &eth->pci_dev->dev, "cdev_add failed");
		return ret;
	}

	dev_dbg( &eth->pci_dev->dev, "eth = 0x%08x, cdev = 0x%08x", (unsigned int)eth, (unsigned int)&eth->cdev);

	return 0;
}

static int remove_character_devices(struct hbm_eth_dev *eth)
{
	BUG_ON(!eth);

	dev_dbg(&eth->pci_dev->dev, "remove_character_devices\n");

	cdev_del(&eth->cdev);
	unregister_chrdev_region(eth->chrdev_region, ETH_CHARDEV_MINOR_SIZE);

	return 0;
}

/*
	PCI DRIVER RELATED SECTION
*/
#define INTEL 0x8086
#define E1000_DEV_ID_82576                    0x10C9
#define E1000_DEV_ID_82576_FIBER              0x10E6
#define E1000_DEV_ID_82576_SERDES             0x10E7
#define E1000_DEV_ID_82576_QUAD_COPPER        0x10E8
#define E1000_DEV_ID_82576_NS                 0x150A
#define E1000_DEV_ID_82576_SERDES_QUAD        0x150D
#define E1000_DEV_ID_82575EB_COPPER           0x10A7
#define E1000_DEV_ID_82575EB_FIBER_SERDES     0x10A9
#define E1000_DEV_ID_82575GB_QUAD_COPPER      0x10D6

static struct pci_device_id hbm_eth_pci_tbl[] = {
	{ PCI_DEVICE(INTEL, E1000_DEV_ID_82576) },
	{ PCI_DEVICE(INTEL, E1000_DEV_ID_82576_NS) },
	{ PCI_DEVICE(INTEL, E1000_DEV_ID_82576_FIBER) },
	{ PCI_DEVICE(INTEL, E1000_DEV_ID_82576_SERDES) },
	{ PCI_DEVICE(INTEL, E1000_DEV_ID_82576_SERDES_QUAD) },
	{ PCI_DEVICE(INTEL, E1000_DEV_ID_82576_QUAD_COPPER) },
	{ PCI_DEVICE(INTEL, E1000_DEV_ID_82575EB_COPPER) },
	{ PCI_DEVICE(INTEL, E1000_DEV_ID_82575EB_FIBER_SERDES) },
	{ PCI_DEVICE(INTEL, E1000_DEV_ID_82575GB_QUAD_COPPER) },
	{0,}
};
MODULE_DEVICE_TABLE (pci, hbm_eth_pci_tbl);

static int hbm_eth_pci_setup(struct pci_dev *pdev, struct hbm_eth_dev *eth)
{
	int ret = -EINVAL;
	unsigned long tmp;

	BUG_ON(!pdev);
	BUG_ON(!eth);

	dev_dbg(&eth->pci_dev->dev, "hbm_eth_pci_setup\n");

	/* Get PCI config info from the Linux kernel (no need to access */
	/* it again via PCI config cycles) */

	eth->eth_mem_base_phys = pci_resource_start(pdev, 0);
	tmp = pci_resource_end(pdev, 0);	/* address is of last valid address */

	eth->eth_mem_size = tmp - eth->eth_mem_base_phys + 1; 
	tmp = pci_resource_flags(pdev, 0);

	BUG_ON(0 == (tmp & IORESOURCE_MEM)); /* This chunk must be MEM */

	eth->eth_mem_base_virt = ioremap(eth->eth_mem_base_phys, eth->eth_mem_size);
	if (! eth->eth_mem_base_virt)
	{
		dev_info( &pdev->dev, "Could not ioremap() eth PCI memory area (phys = 0x%08lx, "
				"size=0x%04x)", eth->eth_mem_base_phys, eth->eth_mem_size);
		goto pci_setup_error;
	}

	return 0;

pci_setup_error:
	if (eth->eth_mem_base_virt)
		iounmap(eth->eth_mem_base_virt);

	return ret;
}


static int __devinit hbm_eth_pci_probe(struct pci_dev *pdev,
				       const struct pci_device_id *ent)
{
	struct hbm_eth_dev *eth = NULL;
	int ret;

	dev_info( &pdev->dev, "hbm_eth_pci_probe(): Found supported Ethernet device 0x%04x", pdev->device);

	/* Now allocate (zeroed) memory for our internal bookkeeping */
	eth = kzalloc( sizeof(struct hbm_eth_dev), GFP_KERNEL);
	if (!eth)
	{
		dev_err( &pdev->dev, "Could not allocate struct hbm_eth_dev, aborting...");
		ret = -ENOMEM;
		goto probe_out;
	}

	pci_set_drvdata (pdev, eth);
	pci_enable_device(pdev); // TODO: check return value;

	ret = pci_request_regions(pdev, MODNAME);
	if (ret)
	{
		dev_err( &pdev->dev, "Could not claim pci device related regions, aborting... (err = %d / 0x%02x)", ret, ret);
		goto probe_out_disable;
	}

	eth->pci_dev = pdev;	
	ret = hbm_eth_pci_setup(pdev, eth);
	if (ret)
	{
		dev_err( &pdev->dev, "hbm_eth_pci_setup() failed, aborting... (err = %d / 0x%02x)", ret, ret);
		goto probe_out_release;
	}

	pci_set_master(pdev);

	ret = setup_character_devices(eth);
	if (ret)
	{
		dev_err( &pdev->dev, "Could not setup character devices, aborting... (err = %d / 0x%02x)", ret, ret);
		goto probe_out_free;
	}

	dev_info( &pdev->dev, "hbm_eth_pci_probe() finished with success.");
	return 0;

probe_out_chardevs:
	remove_character_devices(eth);
probe_out_release:
	pci_release_regions(pdev);
probe_out_disable:
	pci_disable_device(pdev);
probe_out_free:
	if	(eth)
	{
		pci_set_drvdata(pdev, NULL);
		kfree(eth);
	}
probe_out:
	dev_err( &pdev->dev, "hbm_eth_pci_probe() failed... (err = %d / 0x%02x)", ret, ret);

	return ret;
}

static void __devexit hbm_eth_pci_remove(struct pci_dev *pdev)
{
	struct hbm_eth_dev * eth = NULL;

	BUG_ON(!pdev);
	
	eth = pci_get_drvdata(pdev);

	BUG_ON(!eth);

	dev_dbg(&pdev->dev, "hbm_eth_pci_remove\n");
	pci_set_drvdata(pdev, NULL);
	
	remove_character_devices(eth);

	pci_clear_master(pdev);
	pci_release_regions(pdev);
	pci_disable_device(pdev);

	if(eth->eth_mem_base_virt)
		iounmap(eth->eth_mem_base_virt);

	kfree(eth);
}


#ifdef CONFIG_PM
/* 	The following functions are for power management ( PM )	*/

static int hbm_eth_pci_suspend(struct pci_dev *pdev, pm_message_t state)
{
	struct hbm_eth_dev *dev =  pci_get_drvdata(pdev);

	/* TODO: found out what these next 2 lines do */
	pci_save_state(pdev);
	pci_set_power_state(pdev, PCI_D3hot);

	return 0;	/* Never return anything else since it will prevent */	
				/* the kernel from going into suspend mode! */
}

static int hbm_eth_pci_resume(struct pci_dev *pdev)
{
	struct hbm_eth_dev *dev =  pci_get_drvdata(pdev);

	pci_set_power_state(pdev, PCI_D0);
	pci_restore_state(pdev);
	return 0;
}
#endif /* CONFIG_PM */

static struct pci_driver hbm_eth_pci_driver = {
	.name		= MODNAME,
	.id_table	= hbm_eth_pci_tbl,
	.probe		= hbm_eth_pci_probe,
	.remove		= __devexit_p(hbm_eth_pci_remove),
#ifdef CONFIG_PM
	.suspend	= hbm_eth_pci_suspend,
	.resume		= hbm_eth_pci_resume,
#endif /* CONFIG_PM */
};

static int __init hbm_eth_core_init_module(void)
{
	printk(KERN_INFO "HBM Ethernet core Driver: version: %s, Build date: %s %s\n", DRIVER_VERSION, __TIME__, __DATE__);

	/* when a module, this is printed whether or not devices are found in probe */
	return pci_register_driver(&hbm_eth_pci_driver);
}

static void __exit hbm_eth_core_cleanup_module(void)
{
	pci_unregister_driver(&hbm_eth_pci_driver);
}

MODULE_AUTHOR("Nathan Huizinga <nathan.huizinga@chess.nl>");
MODULE_DESCRIPTION("HBM core Ethernet driver");
MODULE_LICENSE("GPL");

module_init(hbm_eth_core_init_module);
module_exit(hbm_eth_core_cleanup_module);
