/*
 * Copyright (C) 2011 HBM Netherlands B.V.
 * Schutweg 15a
 * 5145NP Waalwijk
 * The Netherlands
 * http://www.hbm.com
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/*
 * HBM IntfMod2 FPGA core driver.
 *
 * Handles the FPGA core registration and provide plugin slots
 *
 * The I2C and SPI functions of the fpga are driven by a different
 * drivers. This driver just registers platform devices with associated
 * platform data that contains some pointers to functions to set/get
 * I2C and SPI related functions.
 *
 * This driver will register a character device that can handle five minor numbers:
 *
 * 0 : MsqQueue interface. This device implements the select()/ poll() method, the
 *     file descriptor becomes readable when a message has been received in the
 *     Message Queue. Consecutive reads on the file descriptor will return
 *     consecutive messages, one for every call.
 *
 * 1 : Timebase interrupt. A file descriptor on a device node with minor 1 will become
 *     readable every time a timebase interrupt is received. Reading from the file
 *     descriptor will....<TBD>
 *
 * 2 : CMD_NOW interrupt. A file descriptor on a device node with minor 2 will
 *     become readable every time a COMMAND_NOW interrupt is received. Reading from
 *     the file descriptor will....<TBD>
 *
 * 3 : Compound page
 *
 * 4 : MASTERSLAVE interrupt. A file descriptor on a device node with minor 4 will
 *     become readable every time a MASTERSLAVE interrupt is received. Reading from
 *     the file descriptor will....<TBD>
 *
 */
// enable printing debugging info
// #define DEBUG

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/platform_device.h>
#include <linux/pci.h>
#include <linux/init.h>
#include <linux/ioport.h>
#include <asm/io.h>
#include <linux/interrupt.h>
#include <linux/slab.h>
#include <linux/cdev.h>
#include <linux/poll.h>
#include <linux/bug.h>
#include <linux/gfp.h>
#include <linux/kfifo.h>
#include <linux/wait.h>
#include <linux/sched.h>
#include <linux/wait.h>
#include <linux/mutex.h>


#include "hbm-fpga-core.h"
#include "hbm-fpga.h"

#define MODNAME         "hbm-fpga-core"
#define DRIVER_VERSION  "1.7"

#define EXT_NR_RAM_PAGES        (FPGA_DATA_BUFFER_SIZE >> PAGE_SHIFT)
/*
 * Note:
 * A "normal" compound page exists of at least 2 kernel pages. We use a custom
 * compound page which exists of a single kernel page.
 */
#define EXT_NR_COMPOUND_PAGES   (EXT_NR_RAM_PAGES)

/*
 * Note:
 * This static reference to the fpga device structure and is
 * used by hbm_fpga_pagetable_free_compound_page callback!
 */
static struct hbm_fpga_dev * s_fpga_dev = NULL;


#ifdef ZBUF_CPUSHARED

#define CPUSHARED_NR_RAM_PAGES (CPUSHARED_MEMORY_SIZE  >> PAGE_SHIFT)
#define CPUSHARED_NR_COMPOUND_PAGES (CPUSHARED_NR_RAM_PAGES)
static void hbm_cpushared_pagetable_init(struct hbm_fpga_dev *fpga);
static int hbm_cpushared_pagetable_get_page_params(struct hbm_fpga_dev * fpga, compound_page_params * pcPageParams,
    unsigned int * pPageStart, unsigned int * pPageOffset, unsigned int * pNrPages);
static int hbm_cpushared_mem_fault(struct vm_area_struct *vma, struct vm_fault *vmf);
static int hbm_cpushared_chrdev_mmap(struct file *f, struct vm_area_struct *vma);

#endif /* ZBUF_CPUSHARED */


/*
    I2C RELATED SECTION
*/

#if defined(CONFIG_I2C) || defined(CONFIG_I2C_MODULE)

static void hbm_fpga_i2c_setreg(void * fpga, unsigned int reg, unsigned int data)
{
    struct hbm_fpga_dev *pFpga;
    unsigned int * pData = NULL;
    pFpga = fpga;

    pData = (unsigned int *)(&pFpga->pSystem->I2CPRERL + reg);
    *pData = data;
}

static unsigned int hbm_fpga_i2c_getreg(void * fpga, unsigned int reg)
{
    struct hbm_fpga_dev *pFpga;
    unsigned int * pData = NULL;
    pFpga = fpga;

    pData = (unsigned int *)(&pFpga->pSystem->I2CPRERL + reg);

    return *pData;
}


static int hbm_fpga_i2c_release(struct device * dev)
{
    // dummy function
    return 0;
}

static struct resource ocores_resources[] = {
        [0] = {
               .start  = 0,
               .end    = 0,
               .flags  = IORESOURCE_IRQ,
        },
};

static struct platform_device_id hbm_fpga_i2c_platdev_id = {
    .name           = "hbm-fpga-i2c-platdev",

    /* we have only one type, so no use for this field */
    .driver_data    = 0,
};

static struct platform_device hbm_fpga_i2c_platform_dev  = {
    .name           = "hbm-fpga-i2c-platdev",
    .num_resources  = ARRAY_SIZE(ocores_resources),
    .resource       = ocores_resources,
    .id_entry       = &hbm_fpga_i2c_platdev_id,
    .dev =  {
            .release        = hbm_fpga_i2c_release,
            },
};

/**
 * Fill struct hbm_fpga_dev with the correct pointer to the I2C related
 * toggle functions and register the platform device for the I2C driver.
 * @param fpga pointer to the fpga device bookkeeping structure.
 * @return 0 on success or an negative, non-zero error code otherwise.
*/
static int hbm_fpga_init_i2c(struct hbm_fpga_dev *fpga)
{
    int ret;

    BUG_ON(!fpga);

    dev_dbg(&fpga->pci_dev->dev, "hbm_fpga_init_i2c\n");

    // setup i2c call backs
    fpga->i2c_clock_khz = 50000,       /* input clock of 50MHz */
    fpga->i2c_setreg = hbm_fpga_i2c_setreg;
    fpga->i2c_getreg = hbm_fpga_i2c_getreg;

    hbm_fpga_i2c_platform_dev.dev.platform_data = fpga;

    // IORESOURCE_IRQ
    ocores_resources[0].start = fpga->irq;
    ocores_resources[0].end = ocores_resources[1].start;

    /* register the platform device, the driver will be probe()ed (in hbm_fpga_i2c.c)
     * once loaded
    */
    ret = platform_device_register(&hbm_fpga_i2c_platform_dev);

    dev_dbg(&fpga->pci_dev->dev, "hbm_fpga_init_i2c: ret: %d\n", ret);

    return ret;
}

/**
 * Clean up everything that was claimed by hbm_fpga_init_i2c().
 * @see #hbm_fpga_init_i2c
 * @return 0 on success, negative, non-zero error code otherwise.
*/
static int hbm_fpga_uninit_i2c(struct hbm_fpga_dev *fpga)
{
    BUG_ON(!fpga);

    // disable I2C interrupt
    fpga->pSystem->INTEN &= ~(1 << INT_I2C);

    platform_device_unregister(&hbm_fpga_i2c_platform_dev);
    return 0;
}
#else
static int hbm_fpga_init_i2c(struct hbm_fpga_dev *fpga)
{
    /* Empty function call. Is 'static' so should be
     * optimized out by the compiler...
    */
    return 0;
}

static int hbm_fpga_uninit_i2c(struct hbm_fpga_dev *fpga)
{
    /* Empty function call. Is 'static' so should be
     * optimized out by the compiler...
    */
    return 0;
}
#endif

static void hbm_fpga_pagetable_init(struct hbm_fpga_dev *fpga)
{
    unsigned long phys_addr = fpga->fpga_mem_base_phys + FPGA_DATA_BUFFER_OFFSET;
    struct page * pPage;
    int i;

    dev_dbg(&fpga->pci_dev->dev, "hbm_fpga_pagetable_init");

    /*
     * Initialize page table spinlock.
     */
    spin_lock_init(&fpga->spinlock_pagetable);

    /*
     * Initialize page table:
     * - Mark every page private and owned by us
     * - Pre-calculate physical address of every page
     */
    for (i = 0; i < EXT_NR_COMPOUND_PAGES; i++)
    {
        pPage = &fpga->pExt_ram_page[i];

        // Mark this page as 'special', owned by us (=private ;-).
        SetPageOwnerPriv1(pPage);

        // Set private field containing the physical address of the page
        SetPagePrivate(pPage);
        set_page_private(pPage, phys_addr);

        // Increment to the next physical address
        phys_addr += PAGE_SIZE;
    }
#ifdef ZBUF_CPUSHARED
    hbm_cpushared_pagetable_init (fpga);
#endif /* ZBUF_CPUSHARED */
}

static int hbm_fpga_pagetable_get_page_params(struct hbm_fpga_dev * fpga, compound_page_params * pcPageParams,
    unsigned int * pPageStart, unsigned int * pPageOffset, unsigned int * pNrPages)
{
    int ret = 0;
    unsigned long fpga_ram_base_phys = fpga->fpga_mem_base_phys + FPGA_DATA_BUFFER_OFFSET;

    if ( (pcPageParams->address < fpga_ram_base_phys) ||
         (pcPageParams->size == 0) ||
         ((pcPageParams->address + pcPageParams->size) >= (fpga_ram_base_phys + FPGA_DATA_BUFFER_SIZE)) )
    {
#ifdef ZBUF_CPUSHARED
        return hbm_cpushared_pagetable_get_page_params(fpga, pcPageParams, pPageStart, pPageOffset, pNrPages);
#else /* ZBUF_CPUSHARED */
        ret = -EINVAL;
#endif /* ZBUF_CPUSHARED */
    }
    else
    {
        /*
         * Calculate first page and offset within the page of the requested range.
         * - Subtract physical base address from the range start address
         */
        unsigned int page_start = PFN_DOWN(pcPageParams->address - fpga_ram_base_phys);
        unsigned int page_offset = offset_in_page(pcPageParams->address - fpga_ram_base_phys);
        unsigned int nr_pages = PFN_UP(pcPageParams->size + page_offset);

        /*
         * Sanity check.
         */
        if ( (page_start + nr_pages) >= EXT_NR_RAM_PAGES)
        {
            ret = -EINVAL;
        }
        else
        {
            // Return page parameters
            *pPageStart = page_start;
            *pPageOffset = page_offset;
            *pNrPages = nr_pages;
        }
    }

    return ret;
}

static int hbm_fpga_pagetable_free_compound_page(struct hbm_fpga_dev * fpga, struct page * page_head)
{
    int ret = 1;
    int nr_pages;
    int i;
    struct page * page = page_head + 1;

    /* Get nr of pages belonging to this compound page */
    nr_pages = compound_order(page_head);
#ifdef ZBUF_CPUSHARED
    if ( (nr_pages <= 0) ||
         (nr_pages > EXT_NR_COMPOUND_PAGES + CPUSHARED_NR_COMPOUND_PAGES) )
#else /* ZBUF_CPUSHARED */
    if ( (nr_pages <= 0) ||
         (nr_pages > EXT_NR_COMPOUND_PAGES) )
#endif /* ZBUF_CPUSHARED */
    {
        printk(KERN_ERR "hbm_fpga_pagetable_free_compound_page: invalid nr of pages (%d)!\n", nr_pages);
        return 0;
    }

    /* Lock page table */
    spin_lock(&fpga->spinlock_pagetable);

    /* Clear page head */
    __ClearPageHead(page_head);

    /* Clear remaining page tails belonging to the page head. */
    for (i = 1; i < nr_pages; i++)
    {
        if (compound_head(page) != page_head)
        {
            printk(KERN_ERR "hbm_fpga_pagetable_free_compound_page: invalid tail page (0x%x)!\n", (unsigned int)page);
            ret = 0;
            break;
        }

        /* Clear page tail */
        __ClearPageTail(page);

        /* Clear compound head reference */
        page->first_page = NULL;
        page++;
    }

    /* Unlock page table */
    spin_unlock(&fpga->spinlock_pagetable);

    return ret;
}

/*
 * Note:
 * The page passed into this callback is always a head page!
 */
static void hbm_fpga_pagetable_dtor_compound_page(struct page * page)
{
#ifdef DEBUG
    printk(KERN_DEBUG "hbm_fpga_pagetable_dtor_compound_page: (0x%08x) 0x%08x\n",
        (unsigned int)page, (unsigned int)page_private(page));

    printk(KERN_DEBUG "* Pg offset: 0x%x\n", (unsigned int)page_index(page));
    printk(KERN_DEBUG "* Nr pages : %d\n", compound_order(page));
#endif  // DEBUG

    BUG_ON(!s_fpga_dev);

    if (hbm_fpga_pagetable_free_compound_page(s_fpga_dev, page))
    {
        if (s_fpga_dev->chardev[FPGA_CHARDEV_MINOR_COMPOUND_PAGE].is_active > 0)
        {
            unsigned long mem_phys_addr;

            /* Post physical address in the 'free' queue. */
            mem_phys_addr = (unsigned long)(page_private(page) + page_index(page));

#ifdef DEBUG
            printk(KERN_DEBUG "* Post address = 0x%x\n", (unsigned int)mem_phys_addr);
#endif  // DEBUG

            if (MEM_PHYS_ADDR_SIZE != kfifo_in(s_fpga_dev->chardev[FPGA_CHARDEV_MINOR_COMPOUND_PAGE].fifo_in,
                    (unsigned char *)&mem_phys_addr, MEM_PHYS_ADDR_SIZE))
            {
                printk(KERN_ERR "hbm_fpga_pagetable_dtor_compound_page: kfifo_put failed!\n");
            }
            else
            {
/*                printk(KERN_DEBUG "hbm_fpga_pagetable_dtor_compound_page: waking up wait_queue\n");*/

                wake_up_interruptible(&s_fpga_dev->chardev[FPGA_CHARDEV_MINOR_COMPOUND_PAGE].wait_queue);
            }
        }
    }
}

static int hbm_fpga_pagetable_ctor_compound_page(struct hbm_fpga_dev * fpga, compound_page_params * pcPageParams)
{
    int ret = 0;
    unsigned int page_start = 0;
    unsigned int page_offset = 0;
    unsigned int nr_pages = 0;

    dev_dbg(&fpga->pci_dev->dev, "hbm_fpga_pagetable_ctor_compound_page\n");

    ret = hbm_fpga_pagetable_get_page_params(fpga, pcPageParams, &page_start, &page_offset, &nr_pages);
    if (0 == ret)
    {
        int i;
        struct page * page;
        struct page * page_head;

        /*
         * Log single page requests for testing/analysis purposes.
         */
#ifdef DEBUG
        if (nr_pages == 1)
        {
            dev_info(&fpga->pci_dev->dev, "hbm_fpga_pagetable_ctor_compound_page: single page request.\n");
        }
#endif /* DEBUG */

        /* Lock page table */
        spin_lock(&fpga->spinlock_pagetable);

        if (0 == ret)
        {
            /*
             * Check if the requested range is free, not yet (part of) a compound page.
             */
            for (i = 0; i < nr_pages; i++)
            {
                page = &fpga->pExt_ram_page[page_start + i];

                if (PageCompound(page))
                {
                    dev_warn(&fpga->pci_dev->dev, "Page [0x%04x] not free!\n", (page_start + i));

                    /*
                     * Page is already (part of) a compound page.
                     * Report a "busy" resource error.
                     */
                    ret = -EBUSY;
                    break;
                }

                if (page_count(page) != 0)
                {
                    dev_warn(&fpga->pci_dev->dev, "Page [0x%04x] not released!\n", (page_start + i));

                    /*
                     * Page is still "in use" and not released yet.
                     * Report a "busy" resource error.
                     */
                    ret = -EBUSY;
                    break;
                }
            }
        }

        if (0 == ret)
        {
            /* Range is free, make the compound page. */

            /* Set head page */
            page_head = &fpga->pExt_ram_page[page_start];
            __SetPageHead(page_head);

            /* Set destructor callback function */
            set_compound_page_dtor(page_head, hbm_fpga_pagetable_dtor_compound_page);

            /* Store nr of pages in this compound page */
            set_compound_order(page_head, nr_pages);

            /*
             * Store page offset in this compound page. This is needed to
             * reconstruct the original requested physical address. Because is
             * address is not guaranteed to be page aligned!
             */
            page_head->index = page_offset;

            /*
             * Initialize page reference count to 1 to avoid a race condition
             * which can result in a reference count of 0 between calls at
             * the user level.
             */
            init_page_count(page_head);

//          printk(KERN_DEBUG "! Head Page 0x%04x (0x%08x) [0x%08x] - %d\n",
//              page_start, (unsigned int)page_head, (unsigned int)page_private(page_head) + (unsigned int)page_index(page_head),
//              compound_order(page_head));

            dev_dbg(&fpga->pci_dev->dev, "Head Page 0x%04x (0x%08x) [0x%08x]\n",
                page_start, (unsigned int)page_head, (unsigned int)page_private(page_head));
            dev_dbg(&fpga->pci_dev->dev, "Page count: %d\n", (unsigned int)page_count(page_head));
            dev_dbg(&fpga->pci_dev->dev, "Page offset: 0x%x\n", (unsigned int)page_index(page_head));
            dev_dbg(&fpga->pci_dev->dev, "Compound set: %d\n", (PageCompound(page_head) != 0));
            dev_dbg(&fpga->pci_dev->dev, "Head set: %d\n", PageHead(page_head));
            dev_dbg(&fpga->pci_dev->dev, "Tail set: %d\n", PageTail(page_head));
            dev_dbg(&fpga->pci_dev->dev, "Nr pages: %d\n", compound_order(page_head));
            dev_dbg(&fpga->pci_dev->dev, "Compound head: 0x%x\n", (unsigned int)compound_head(page_head));

            /* Set remaining tail pages */
            for (i = 1; i < nr_pages; i++)
            {
                page = &fpga->pExt_ram_page[page_start + i];
                __SetPageTail(page);
                page->first_page = page_head;

                dev_dbg(&fpga->pci_dev->dev, "Tail Page 0x%04x (0x%08x) [0x%08x]",
                    page_start + i, (unsigned int)page, (unsigned int)page_private(page));
                dev_dbg(&fpga->pci_dev->dev, "Compound set: %d", PageCompound(page));
                dev_dbg(&fpga->pci_dev->dev, "Compound head: 0x%x\n", (unsigned int)compound_head(page));
                dev_dbg(&fpga->pci_dev->dev, "Head set: %d", PageHead(page));
                dev_dbg(&fpga->pci_dev->dev, "Tail set: %d", PageTail(page));
            }
        }

        /* Unlock page table */
        spin_unlock(&fpga->spinlock_pagetable);
    }

    return ret;
}

static int hbm_fpga_pagetable_test_get_page(struct hbm_fpga_dev * fpga, compound_page_params * pcPageParams)
{
    int ret = 0;
    unsigned int page_start = 0;
    unsigned int page_offset = 0;
    unsigned int nr_pages = 0;

    dev_dbg(&fpga->pci_dev->dev, "hbm_fpga_pagetable_test_get_page:\n");

    ret = hbm_fpga_pagetable_get_page_params(fpga, pcPageParams, &page_start, &page_offset, &nr_pages);
    if (0 == ret)
    {
        int i;
        struct page * page;

        for (i = 0; i < nr_pages; i++)
        {
            page = &fpga->pExt_ram_page[page_start + i];

            dev_dbg(&fpga->pci_dev->dev, "Page 0x%04x (0x%08x) [0x%08x]\n",
                page_start + i, (unsigned int)page, (unsigned int)page_private(page));
            if (!PageCompound(page))
            {
                dev_dbg(&fpga->pci_dev->dev, "Error: this is not a compound page!\n");
                ret = -EINVAL;
                break;
            }

            dev_dbg(&fpga->pci_dev->dev, "Head set : %d\n", PageHead(page));
            dev_dbg(&fpga->pci_dev->dev, "Tail set : %d\n", PageTail(page));
            dev_dbg(&fpga->pci_dev->dev, "Count    : %d\n", page_count(page));
            dev_dbg(&fpga->pci_dev->dev, "Pg offset: 0x%x\n", (unsigned int)page_index(page));

            /* Call: get page to increment page counter */
            get_page(page);

            dev_dbg(&fpga->pci_dev->dev, "Post count: %d\n", page_count(page));
        }
    }

    return ret;
}

static int hbm_fpga_pagetable_test_put_page(struct hbm_fpga_dev * fpga, compound_page_params * pcPageParams)
{
    int ret = 0;
    unsigned int page_start = 0;
    unsigned int page_offset = 0;
    unsigned int nr_pages = 0;

    dev_dbg(&fpga->pci_dev->dev, "hbm_fpga_pagetable_test_put_page:\n");

    ret = hbm_fpga_pagetable_get_page_params(fpga, pcPageParams, &page_start, &page_offset, &nr_pages);
    if (0 == ret)
    {
        int i;
        struct page * page;

        for (i = 0; i < nr_pages; i++)
        {
            page = &fpga->pExt_ram_page[page_start + i];

            dev_dbg(&fpga->pci_dev->dev, "Page 0x%04x (0x%08x) [0x%08x]\n",
                page_start + i, (unsigned int)page, (unsigned int)page_private(page));
            if (!PageCompound(page))
            {
                dev_dbg(&fpga->pci_dev->dev, "Error: this is not a compound page!\n");
                ret = -EINVAL;
                break;
            }

            dev_dbg(&fpga->pci_dev->dev, "Head set : %d\n", PageHead(page));
            dev_dbg(&fpga->pci_dev->dev, "Tail set : %d\n", PageTail(page));
            dev_dbg(&fpga->pci_dev->dev, "Count    : %d\n", page_count(page));
            dev_dbg(&fpga->pci_dev->dev, "Pg offset: 0x%x\n", (unsigned int)page_index(page));

            /* Call: put page to decrement page counter */
            put_page(page);
        }
    }

    return ret;
}

static int hbm_fpga_pagetable_cleanup_compound_page(struct hbm_fpga_dev * fpga, compound_page_params * pcPageParams)
{
    int ret = 0;
    unsigned int page_start = 0;
    unsigned int page_offset = 0;
    unsigned int nr_pages = 0;

    dev_dbg(&fpga->pci_dev->dev, "hbm_fpga_pagetable_cleanup_compound_page:\n");

    ret = hbm_fpga_pagetable_get_page_params(fpga, pcPageParams, &page_start, &page_offset, &nr_pages);
    if (0 == ret)
    {
        struct page * page = &fpga->pExt_ram_page[page_start];

        dev_dbg(&fpga->pci_dev->dev, "Page 0x%04x (0x%08x) [0x%08x]\n",
            page_start, (unsigned int)page, (unsigned int)page_private(page));

        if ( PageCompound(page) &&
             (get_compound_page_dtor(page) == hbm_fpga_pagetable_dtor_compound_page) )
        {
            dev_dbg(&fpga->pci_dev->dev, "Pg offset: 0x%x\n", (unsigned int)page_index(page));
            dev_dbg(&fpga->pci_dev->dev, "Nr pages : %d\n", compound_order(page));

            if (!hbm_fpga_pagetable_free_compound_page(fpga, page))
            {
                dev_dbg(&fpga->pci_dev->dev, "Error freeing compound page!\n");
                ret = -EINVAL;
            }
        }
        else
        {
            dev_dbg(&fpga->pci_dev->dev, "Error: this is not a/our compound page!\n");
            ret = -EINVAL;
        }
    }

    return ret;
}

static int hbm_fpga_pagetable_unlock_compound_page(struct hbm_fpga_dev * fpga, compound_page_params * pcPageParams)
{
    int ret = 0;
    unsigned int page_start = 0;
    unsigned int page_offset = 0;
    unsigned int nr_pages = 0;

    dev_dbg(&fpga->pci_dev->dev, "hbm_fpga_pagetable_unlock_compound_page:\n");

    ret = hbm_fpga_pagetable_get_page_params(fpga, pcPageParams, &page_start, &page_offset, &nr_pages);
    if (0 == ret)
    {
        struct page * page = &fpga->pExt_ram_page[page_start];

        dev_dbg(&fpga->pci_dev->dev, "Page 0x%04x (0x%08x) [0x%08x]\n",
            page_start, (unsigned int)page, (unsigned int)page_private(page));

        if ( PageCompound(page) &&
             (get_compound_page_dtor(page) == hbm_fpga_pagetable_dtor_compound_page) )
        {
            dev_dbg(&fpga->pci_dev->dev, "Page count: 0x%x\n", (unsigned int)page_count(page));
            dev_dbg(&fpga->pci_dev->dev, "Page offset: 0x%x\n", (unsigned int)page_index(page));
            dev_dbg(&fpga->pci_dev->dev, "Nr pages : %d\n", compound_order(page));

            /*
             * Put page to decrement page counter.
             * This will allow the page reference counter to reach 0 and be
             * freed.
             */
            put_page(page);
        }
        else
        {
            dev_dbg(&fpga->pci_dev->dev, "Error: this is not a/our compound page!\n");
            ret = -EINVAL;
        }
    }

    return ret;
}

/*
    CHARACTER DEVICES SECTION
*/

#define SWITCH_MINOR(i) switch(iminor(i))

/*
 * read frame offset of oldest message in FIFO
 */

static ssize_t hbm_fpga_chrdev_read(struct file *f, char __user *buffer, size_t count, loff_t *offset)
{
    int                     ret = -EINVAL;
    struct hbm_fpga_dev *   fpga;
    struct kfifo *          pFifo;
    unsigned int            frameOffset;
    unsigned int            tcdr;
    unsigned int            msmsgstat;
    unsigned long           mem_phys_addr;
    TIMEBASE_STRUCT         timebase_struct;
    int                     minor;
    int                     len = 0;

    BUG_ON(!f);

    fpga = (struct hbm_fpga_dev *) f->private_data;
    BUG_ON(!fpga);

    // Get minor device number
    minor = iminor(f->f_dentry->d_inode);
//  dev_dbg(&fpga->pci_dev->dev, "hbm_fpga_chrdev_read: minor %d\n", minor);

    // Check parameters, ignore offset
    if (buffer == NULL)
    {
        dev_err(&fpga->pci_dev->dev, "hbm_fpga_chrdev_read: buffer: 0x%x, count: %d, offset: %d\n", (unsigned int)buffer, count, (int)offset);
        return -EFAULT;
    }

    // Check blocking
    if (f->f_flags & O_NONBLOCK)
    {
        // only blocking mode supported
        return -EAGAIN;
    }

//  dev_dbg(&fpga->pci_dev->dev, "hbm_fpga_chrdev_read\n");
    switch (minor)
    {
        case FPGA_CHARDEV_MINOR_MSG_QUEUE:

/*            fpga->pDebug->SCRATCH1 = 0x30;*/

            // lock
//          dev_dbg(&fpga->pci_dev->dev, "hbm_fpga_chrdev_read: MSG_QUEUE\n");

//temp disabled         spin_lock(fpga->chardev[FPGA_CHARDEV_MINOR_MSG_QUEUE].spinlock_read);

            // Check parameters, ignore offset
            if (count != FRAME_OFFSET_SIZE)
            {
                dev_err(&fpga->pci_dev->dev, "hbm_fpga_chrdev_read: buffer: 0x%x, count: %d, offset: %d\n", (unsigned int)buffer, count, (int)offset);
                return -EFAULT;
            }

            pFifo = fpga->chardev[FPGA_CHARDEV_MINOR_MSG_QUEUE].fifo_in;
            BUG_ON(!pFifo);

/*            fpga->pDebug->SCRATCH1 = 0x31;*/

            // Get entry from FIFO
            if (kfifo_out(pFifo, (unsigned char *)&frameOffset, FRAME_OFFSET_SIZE) == FRAME_OFFSET_SIZE)
            {
/*                fpga->pDebug->SCRATCH1 = 0x32;*/

                // Check frameOffset, out of range
//              if ((frameOffset < FPGA_MU_MEMORY_OFFSET) ||
//                  (frameOffset >= (FPGA_MU_MEMORY_OFFSET + FPGA_MU_MEMORY_SIZE)))
//              {
//                  dev_err(&fpga->pci_dev->dev, "hbm_fpga_chrdev_read: frameOffset out of range: frameOffset0x%x\n", (unsigned int)frameOffset);
//              }

                // Check frameOffset, not multiple of size
                if ((frameOffset % MSQ_MESSAGE_SIZE_BYTES) != 0)
                {
                    dev_err(&fpga->pci_dev->dev, "hbm_fpga_chrdev_read: frameOffset no multiple of msg size: frameOffset0x%x\n", (unsigned int)frameOffset);
                }

                // Copy frame_offset to user space
                if (copy_to_user(buffer, &frameOffset, FRAME_OFFSET_SIZE) != 0)
                {
                    dev_err(&fpga->pci_dev->dev, "hbm_fpga_chrdev_read: copy_to_user failed\n");
                    ret = -EFAULT;
                }
                else
                {
                    // return number of bytes read
                    ret = FRAME_OFFSET_SIZE;
                }

/*                fpga->pDebug->SCRATCH1 = 0x33;*/
            }
            else
            {
/*                fpga->pDebug->SCRATCH1 = 0x35;*/

//              dev_dbg(&fpga->pci_dev->dev, "hbm_fpga_chrdev_read: no data in FIFO, block and wait on waitqueue\n");

                // Block till FRAME_OFFSET_SIZE bytes are in FIFO or process is interrupted
                while (kfifo_len(pFifo) < FRAME_OFFSET_SIZE)
                {
                    if (wait_event_interruptible(fpga->chardev[FPGA_CHARDEV_MINOR_MSG_QUEUE].wait_queue, kfifo_len(pFifo) > 0) != 0)
                    {
/*                        fpga->pDebug->SCRATCH1 = 0x36;*/

                        dev_dbg(&fpga->pci_dev->dev, "hbm_fpga_chrdev_read: wait_event interrupted\n");

                        // Interrupted, return -EINTR to user program
                        return -ERESTARTSYS;
                    }

/*                    fpga->pDebug->SCRATCH1 = 0x3A;*/

//                  dev_dbg(&fpga->pci_dev->dev, "hbm_fpga_chrdev_read: received event\n");
                }

/*                fpga->pDebug->SCRATCH1 = 0x37;*/

//              dev_dbg(&fpga->pci_dev->dev, "hbm_fpga_chrdev_read: copying to user space\n");

                len = kfifo_out(pFifo, (unsigned char *)&frameOffset, FRAME_OFFSET_SIZE);

//              dev_dbg(&fpga->pci_dev->dev, "hbm_fpga_chrdev_read: kfifo_get: len: %d\n", len);

                // Check frameOffset, out of range
//              if ((frameOffset < FPGA_MU_MEMORY_OFFSET) ||
//                  (frameOffset >= (FPGA_MU_MEMORY_OFFSET + FPGA_MU_MEMORY_SIZE)))
//              {
//                  dev_err(&fpga->pci_dev->dev, "hbm_fpga_chrdev_read: frameOffset out of range: frameOffset0x%x\n", (unsigned int)frameOffset);
//              }

                // Check frameOffset, not multiple of size
                if ((frameOffset % MSQ_MESSAGE_SIZE_BYTES) != 0)
                {
                    dev_err(&fpga->pci_dev->dev, "hbm_fpga_chrdev_read: frameOffset no multiple of msg size: frameOffset0x%x\n", (unsigned int)frameOffset);
                }

/*                fpga->pDebug->SCRATCH1 = 0x38;*/

                // Copy frame_offset to user space
                if (copy_to_user(buffer, &frameOffset, FRAME_OFFSET_SIZE) != 0)
                {
                    dev_err(&fpga->pci_dev->dev, "hbm_fpga_chrdev_read: copy_to_user failed\n");
                    ret = -EFAULT;
                }
                else
                {
//                  dev_dbg(&fpga->pci_dev->dev, "hbm_fpga_chrdev_read: copy_to_user: frame offset: 0x%x\n", frameOffset);

                    // return number of bytes read
                    ret = FRAME_OFFSET_SIZE;
                }

/*                fpga->pDebug->SCRATCH1 = 0x39;*/
            }

            // unlock
//temp disabled         spin_unlock(fpga->chardev[FPGA_CHARDEV_MINOR_MSG_QUEUE].spinlock_read);

            break;

            case FPGA_CHARDEV_MINOR_CMD_NOW:

//              dev_dbg(&fpga->pci_dev->dev, "hbm_fpga_chrdev_read: CMD_NOW\n");

                // lock
//              spin_lock(fpga->chardev[FPGA_CHARDEV_MINOR_CMD_NOW].spinlock_read);

                // Check parameters, ignore offset
                if (count != TCDR_SIZE)
                {
                    dev_err(&fpga->pci_dev->dev, "hbm_fpga_chrdev_read: buffer: 0x%x, count: %d, offset: %d\n", (unsigned int)buffer, count, (int)offset);
                    return -EFAULT;
                }

                pFifo = fpga->chardev[FPGA_CHARDEV_MINOR_CMD_NOW].fifo_in;
                BUG_ON(!pFifo);

                // Get entry from FIFO
                if (kfifo_out(pFifo, (unsigned char *)&tcdr, TCDR_SIZE) == TCDR_SIZE)
                {
                    // Copy frame_offset to user space
                    if (copy_to_user(buffer, &tcdr, TCDR_SIZE) != 0)
                    {
                        dev_err(&fpga->pci_dev->dev, "hbm_fpga_chrdev_read: copy_to_user failed\n");
                        ret = -EFAULT;
                    }
                    else
                    {
                        // return number of bytes read
                        ret = TCDR_SIZE;
                    }
                }
                else
                {
//                  dev_dbg(&fpga->pci_dev->dev, "hbm_fpga_chrdev_read: no data in FIFO, block and wait on waitqueue\n");

                    // Block till 4 bytes are in FIFO or process is interrupted
                    while (kfifo_len(pFifo) < TCDR_SIZE)
                    {
                        if (wait_event_interruptible(fpga->chardev[FPGA_CHARDEV_MINOR_CMD_NOW].wait_queue, kfifo_len(pFifo) > 0) != 0)
                        {
                            dev_dbg(&fpga->pci_dev->dev, "hbm_fpga_chrdev_read: wait_event interrupted\n");

                            // Interrupted, return -EINTR to user program
                            return -ERESTARTSYS;
                        }

//                      dev_dbg(&fpga->pci_dev->dev, "hbm_fpga_chrdev_read: received event\n");
                    }

//                  dev_dbg(&fpga->pci_dev->dev, "hbm_fpga_chrdev_read: copying to user space\n");

                    len = kfifo_out(pFifo, (unsigned char *)&tcdr, TCDR_SIZE);

//                  dev_dbg(&fpga->pci_dev->dev, "hbm_fpga_chrdev_read: kfifo_get: len: %d\n", len);

                    // Copy frame_offset to user space
                    if (copy_to_user(buffer, &tcdr, TCDR_SIZE) != 0)
                    {
                        dev_err(&fpga->pci_dev->dev, "hbm_fpga_chrdev_read: copy_to_user failed\n");
                        ret = -EFAULT;
                    }
                    else
                    {
//                      dev_dbg(&fpga->pci_dev->dev, "hbm_fpga_chrdev_read: copy_to_user: tcdrt: 0x%x\n", tcdr);

                        // return number of bytes read
                        ret = TCDR_SIZE;
                    }
                }
                // unlock
//              spin_unlock(fpga->chardev[FPGA_CHARDEV_MINOR_CMD_NOW].spinlock_read);

                break;

        case FPGA_CHARDEV_MINOR_TIMEBASE:

//          dev_dbg(&fpga->pci_dev->dev, "hbm_fpga_chrdev_read: TIMEBASE\n");

            // lock
//          spin_lock(fpga->chardev[FPGA_CHARDEV_MINOR_TIMEBASE].spinlock_read);

            // Check parameters, ignore offset
            if (count != sizeof(TIMEBASE_STRUCT))
            {
                dev_err(&fpga->pci_dev->dev, "hbm_fpga_chrdev_read: buffer: 0x%x, count: %d, offset: %d\n", (unsigned int)buffer, count, (int)offset);
                return -EFAULT;
            }

            pFifo = fpga->chardev[FPGA_CHARDEV_MINOR_TIMEBASE].fifo_in;
            BUG_ON(!pFifo);

            // Get entry from FIFO
            if (kfifo_out(pFifo, (unsigned char *)&timebase_struct, sizeof(TIMEBASE_STRUCT)) == sizeof(TIMEBASE_STRUCT))
            {
                // Copy frame_offset to user space
                if (copy_to_user(buffer, &timebase_struct, sizeof(TIMEBASE_STRUCT)) != 0)
                {
                    dev_err(&fpga->pci_dev->dev, "hbm_fpga_chrdev_read: copy_to_user failed\n");
                    ret = -EFAULT;
                }
                else
                {
                    // return number of bytes read
                    ret = sizeof(TIMEBASE_STRUCT);
                }
            }
            else
            {
//              dev_dbg(&fpga->pci_dev->dev, "hbm_fpga_chrdev_read: no data in FIFO, block and wait on waitqueue\n");

                // Block till TIMEBASE_STRUCT bytes are in FIFO or process is interrupted
                while(kfifo_len(pFifo) < sizeof(TIMEBASE_STRUCT))
                {
                    if (wait_event_interruptible(fpga->chardev[FPGA_CHARDEV_MINOR_TIMEBASE].wait_queue, kfifo_len(pFifo) > 0) != 0)
                    {
                        dev_dbg(&fpga->pci_dev->dev, "hbm_fpga_chrdev_read: wait_event interrupted\n");

                        // Interrupted, return -EINTR to user program
                        return -ERESTARTSYS;
                    }

//                  dev_dbg(&fpga->pci_dev->dev, "hbm_fpga_chrdev_read: received event\n");
                }

//              dev_dbg(&fpga->pci_dev->dev, "hbm_fpga_chrdev_read: copying to user space\n");

                len = kfifo_out(pFifo, (unsigned char *)&timebase_struct, sizeof(TIMEBASE_STRUCT));

//              dev_dbg(&fpga->pci_dev->dev, "hbm_fpga_chrdev_read: kfifo_get: len: %d\n", len);

                // Copy frame_offset to user space
                if(copy_to_user(buffer, &timebase_struct, sizeof(TIMEBASE_STRUCT)) != 0)
                {
                    dev_err(&fpga->pci_dev->dev, "hbm_fpga_chrdev_read: copy_to_user failed\n");
                    ret = -EFAULT;
                }
                else
                {
//                  dev_dbg(&fpga->pci_dev->dev, "hbm_fpga_chrdev_read: copy_to_user: TSR: 0x%x, TS0R: 0x%x, TIDR: 0x%x\n",
//                            timebase_struct.TSR, timebase_struct.TS0R, timebase_struct.TIDR);

                    // return number of bytes read
                    ret = sizeof(TIMEBASE_STRUCT);
                }
            }

            // unlock
//          spin_unlock(fpga->chardev[FPGA_CHARDEV_MINOR_TIMEBASE].spinlock_read);

            break;

        case FPGA_CHARDEV_MINOR_COMPOUND_PAGE:

//              dev_dbg(&fpga->pci_dev->dev, "hbm_fpga_chrdev_read: COMPOUND_PAGE\n");

                // lock
//              spin_lock(fpga->chardev[FPGA_CHARDEV_MINOR_COMPOUND_PAGE].spinlock_read);

                // Check parameters, ignore offset
                if (count != MEM_PHYS_ADDR_SIZE)
                {
                    dev_err(&fpga->pci_dev->dev, "hbm_fpga_chrdev_read: buffer: 0x%x, count: %d, offset: %d\n", (unsigned int)buffer, count, (int)offset);
                    return -EFAULT;
                }

                pFifo = fpga->chardev[FPGA_CHARDEV_MINOR_COMPOUND_PAGE].fifo_in;
                BUG_ON(!pFifo);

                // Get entry from FIFO
                if (kfifo_out(pFifo, (unsigned char *)&mem_phys_addr, MEM_PHYS_ADDR_SIZE) == MEM_PHYS_ADDR_SIZE)
                {
                    // Copy frame_offset to user space
                    if (copy_to_user(buffer, &mem_phys_addr, MEM_PHYS_ADDR_SIZE) != 0)
                    {
                        dev_err(&fpga->pci_dev->dev, "hbm_fpga_chrdev_read: copy_to_user failed\n");
                        ret = -EFAULT;
                    }
                    else
                    {
                        // return number of bytes read
                        ret = MEM_PHYS_ADDR_SIZE;
                    }
                }
                else
                {
//                  dev_dbg(&fpga->pci_dev->dev, "hbm_fpga_chrdev_read: no data in FIFO, block and wait on waitqueue\n");

                    // Block till 4 bytes are in FIFO or process is interrupted
                    while (kfifo_len(pFifo) < MEM_PHYS_ADDR_SIZE)
                    {
                        if (wait_event_interruptible(fpga->chardev[FPGA_CHARDEV_MINOR_COMPOUND_PAGE].wait_queue, kfifo_len(pFifo) > 0) != 0)
                        {
                            dev_dbg(&fpga->pci_dev->dev, "hbm_fpga_chrdev_read: wait_event interrupted\n");

                            // Interrupted, return -EINTR to user program
                            return -ERESTARTSYS;
                        }

//                      dev_dbg(&fpga->pci_dev->dev, "hbm_fpga_chrdev_read: received event\n");
                    }

//                  dev_dbg(&fpga->pci_dev->dev, "hbm_fpga_chrdev_read: copying to user space\n");

                    len = kfifo_out(pFifo, (unsigned char *)&mem_phys_addr, MEM_PHYS_ADDR_SIZE);

//                  dev_dbg(&fpga->pci_dev->dev, "hbm_fpga_chrdev_read: kfifo_get: len: %d\n", len);

                    // Copy frame_offset to user space
                    if (copy_to_user(buffer, &mem_phys_addr, MEM_PHYS_ADDR_SIZE) != 0)
                    {
                        dev_err(&fpga->pci_dev->dev, "hbm_fpga_chrdev_read: copy_to_user failed\n");
                        ret = -EFAULT;
                    }
                    else
                    {
//                      dev_dbg(&fpga->pci_dev->dev, "hbm_fpga_chrdev_read: copy_to_user: mem_phys_addr: 0x%x\n", mem_phys_addr);

                        // return number of bytes read
                        ret = MEM_PHYS_ADDR_SIZE;
                    }
                }
                // unlock
//              spin_unlock(fpga->chardev[FPGA_CHARDEV_MINOR_COMPOUND_PAGE].spinlock_read);

                break;

            case FPGA_CHARDEV_MINOR_MASTERSLAVE:

//              dev_dbg(&fpga->pci_dev->dev, "hbm_fpga_chrdev_read: MASTERSLAVE\n");

                // lock
//              spin_lock(fpga->chardev[FPGA_CHARDEV_MINOR_MASTERSLAVE].spinlock_read);

                // Check parameters, ignore offset
                if (count != MSMSGSTAT_SIZE)
                {
                    dev_err(&fpga->pci_dev->dev, "hbm_fpga_chrdev_read: buffer: 0x%x, count: %d, offset: %d\n", (unsigned int)buffer, count, (int)offset);
                    return -EFAULT;
                }

                pFifo = fpga->chardev[FPGA_CHARDEV_MINOR_MASTERSLAVE].fifo_in;
                BUG_ON(!pFifo);

                // Get entry from FIFO
                if (kfifo_out(pFifo, (unsigned char *)&msmsgstat, MSMSGSTAT_SIZE) == MSMSGSTAT_SIZE)
                {
                    // Copy frame_offset to user space
                    if (copy_to_user(buffer, &msmsgstat, MSMSGSTAT_SIZE) != 0)
                    {
                        dev_err(&fpga->pci_dev->dev, "hbm_fpga_chrdev_read: copy_to_user failed\n");
                        ret = -EFAULT;
                    }
                    else
                    {
                        // return number of bytes read
                        ret = MSMSGSTAT_SIZE;
                    }
                }
                else
                {
//                  dev_dbg(&fpga->pci_dev->dev, "hbm_fpga_chrdev_read: no data in FIFO, block and wait on waitqueue\n");

                    // Block till 4 bytes are in FIFO or process is interrupted
                    while (kfifo_len(pFifo) < MSMSGSTAT_SIZE)
                    {
                        if (wait_event_interruptible(fpga->chardev[FPGA_CHARDEV_MINOR_MASTERSLAVE].wait_queue, kfifo_len(pFifo) > 0) != 0)
                        {
                            dev_dbg(&fpga->pci_dev->dev, "hbm_fpga_chrdev_read: wait_event interrupted\n");

                            // Interrupted, return -EINTR to user program
                            return -ERESTARTSYS;
                        }

//                      dev_dbg(&fpga->pci_dev->dev, "hbm_fpga_chrdev_read: received event\n");
                    }

//                  dev_dbg(&fpga->pci_dev->dev, "hbm_fpga_chrdev_read: copying to user space\n");

                    len = kfifo_out(pFifo, (unsigned char *)&msmsgstat, MSMSGSTAT_SIZE);

//                  dev_dbg(&fpga->pci_dev->dev, "hbm_fpga_chrdev_read: kfifo_get: len: %d\n", len);

                    // Copy frame_offset to user space
                    if (copy_to_user(buffer, &msmsgstat, MSMSGSTAT_SIZE) != 0)
                    {
                        dev_err(&fpga->pci_dev->dev, "hbm_fpga_chrdev_read: copy_to_user failed\n");
                        ret = -EFAULT;
                    }
                    else
                    {
//                      dev_dbg(&fpga->pci_dev->dev, "hbm_fpga_chrdev_read: copy_to_user: msmsgstat: 0x%x\n", msmsgstat);

                        // return number of bytes read
                        ret = MSMSGSTAT_SIZE;
                    }
                }
                // unlock
//              spin_unlock(fpga->chardev[FPGA_CHARDEV_MINOR_MASTERSLAVE].spinlock_read);

                break;

            default:
                /* Invalid minor, reject read() */
                ret = -EINVAL;

                break;
        }

    return ret;
}

/*
 *  Recylces message queue frameOffset, one frameOffset per call
 */
static ssize_t hbm_fpga_chrdev_write(struct file *f, const char __user *buffer, size_t count, loff_t *offset)
{
    int                     ret = -EINVAL;
    struct  hbm_fpga_dev *  fpga;
    unsigned int            frameOffset;
    int                     minor;
    int                     frameOffsetOK = 1;

    BUG_ON(!f);
    fpga = (struct hbm_fpga_dev *) f->private_data;

    BUG_ON(!fpga);
    BUG_ON(!fpga->pFPGAMsgQ);

    minor = iminor(f->f_dentry->d_inode);
//  dev_dbg(&fpga->pci_dev->dev, "hbm_fpga_chrdev_write: minor %d\n", minor);

    // Check parameters
    if ((buffer == NULL) || (count != FRAME_OFFSET_SIZE))
    {
        return -EFAULT;
    }

//  dev_dbg(&fpga->pci_dev->dev, "hbm_fpga_chrdev_write\n");
    switch (minor)
    {
        case FPGA_CHARDEV_MINOR_MSG_QUEUE:
//          fpga->pDebug->SCRATCH1 = 0x40;

            // lock
            //temp disabled         spin_lock(fpga->chardev[FPGA_CHARDEV_MINOR_MSG_QUEUE].spinlock_write);

//          dev_dbg(&fpga->pci_dev->dev, "hbm_fpga_chrdev_write: MSG_QUEUE\n");

            // copy frame offset
            if (copy_from_user(&frameOffset, buffer, FRAME_OFFSET_SIZE) == 0)
            {
//              dev_dbg(&fpga->pci_dev->dev, "hbm_fpga_chrdev_write: frameOffset: 0x%x\n", frameOffset);

                // Check frameOffset, out of range
//              if ((frameOffset < FPGA_MU_MEMORY_OFFSET) ||
//                  (frameOffset >= (FPGA_MU_MEMORY_OFFSET + FPGA_MU_MEMORY_SIZE)))
//              {
//                  fpga->pDebug->SCRATCH1 = 0x43;
//
//                  dev_err(&fpga->pci_dev->dev, "hbm_fpga_chrdev_write: frameOffset out of range: frameOffset0x%x\n", (unsigned int)frameOffset);
//                  frameOffsetOK = 0;
//              }

                // Check frameOffset, not multiple of size
                if ((frameOffset % MSQ_MESSAGE_SIZE_BYTES) != 0)
                {
//                  fpga->pDebug->SCRATCH1 = 0x44;

                    dev_err(&fpga->pci_dev->dev, "hbm_fpga_chrdev_write: frameOffset no multiple of msg size: frameOffset0x%x\n", (unsigned int)frameOffset);
                    frameOffsetOK = 0;
                }

                if (0 != frameOffsetOK)
                {
//                  fpga->pDebug->SCRATCH1 = 0x41;

                    // recycle frame offset by writing to
                    fpga->pFPGAMsgQ->recvMsgQ = frameOffset;

                    // return number of bytes written
                    ret = FRAME_OFFSET_SIZE;

//                  fpga->pDebug->SCRATCH1 = 0x42;
                }
                else
                {
//                  fpga->pDebug->SCRATCH1 = 0x45;

                    // Send "empty" indicator
                    fpga->pFPGAMsgQ->recvMsgQ = 0xffffffff;

//                  fpga->pDebug->SCRATCH1 = 0x46;
                }
            }
            else
            {
                dev_err(&fpga->pci_dev->dev, "hbm_fpga_chrdev_write: error copying data from user space\n");
                ret = -EFAULT;
            }

            // unlock
            //temp disabled         spin_unlock(fpga->chardev[FPGA_CHARDEV_MINOR_MSG_QUEUE].spinlock_write);

            break;

        default:
            /* Invalid minor, reject write() */
            ret = -EINVAL;

            break;
    }

    return ret;
}

static int hbm_fpga_chrdev_ioctl(struct file *f, unsigned int cmd, unsigned long arg)
{
    int ret = -EINVAL;
    struct hbm_fpga_dev *fpga = NULL;
    compound_page_params cPageParams;
    struct inode *i = f->f_dentry->d_inode ;

    fpga = (struct hbm_fpga_dev*) container_of(i->i_cdev, struct hbm_fpga_dev, cdev);

    BUG_ON(!fpga);

// TODO: implement disable, reset, enable

    mutex_lock(&(fpga->driver_mutex) ) ;


    switch (cmd)
    {
        case HBM_FPGA_IOCTL_READ_PHYSICAL_ADDRESS:

//          dev_dbg(&fpga->pci_dev->dev, "hbm_fpga_chrdev_ioctl: HBM_FPGA_IOCTL_READ_PHYSICAL_ADDRESS\n");

            if (copy_to_user((void *)arg, (void *)&(fpga->fpga_mem_base_phys), sizeof(int)) != 0)
            {
                // copy failed
                ret = -EFAULT;
            }
            else
            {
                ret = 0;
            }
            break;

        case HBM_FPGA_IOCTL_SET_TRACE_FLAGS:
//          dev_dbg(&fpga->pci_dev->dev, "hbm_fpga_chrdev_ioctl: HBM_FPGA_IOCTL_SET_TRACE_FLAGS\n");

            if (copy_from_user((void *)&(fpga->traceFlags), (void *)arg, sizeof(unsigned int)) != 0)
            {
                // copy failed
                ret = -EFAULT;
            }
            else
            {
                ret = 0;
            }

            dev_dbg(&fpga->pci_dev->dev, "hbm_fpga_chrdev_ioctl: HBM_FPGA_IOCTL_SET_TRACE_FLAGS: 0x%08x\n", fpga->traceFlags);
            break;

        case HBM_FPGA_IOCTL_MAKE_COMPOUND_PAGE:
//          dev_dbg(&fpga->pci_dev->dev, "hbm_fpga_chrdev_ioctl: HBM_FPGA_IOCTL_MAKE_COMPOUND_PAGE\n");

            if (copy_from_user((void *)&(cPageParams), (void *)arg, sizeof(compound_page_params)) != 0)
            {
                // copy failed
                ret = -EFAULT;
            }
            else
            {
                ret = hbm_fpga_pagetable_ctor_compound_page(fpga, &cPageParams);
            }
            break;

        case HBM_FPGA_IOCTL_CLEANUP_COMPOUND_PAGE:
//          dev_dbg(&fpga->pci_dev->dev, "hbm_fpga_chrdev_ioctl: HBM_FPGA_IOCTL_CLEANUP_COMPOUND_PAGE\n");

            if (copy_from_user((void *)&(cPageParams), (void *)arg, sizeof(compound_page_params)) != 0)
            {
                // copy failed
                ret = -EFAULT;
            }
            else
            {
                ret = hbm_fpga_pagetable_cleanup_compound_page(fpga, &cPageParams);
            }
            break;

        case HBM_FPGA_IOCTL_TEST_GET_PAGE:
//          dev_dbg(&fpga->pci_dev->dev, "hbm_fpga_chrdev_ioctl: HBM_FPGA_IOCTL_TEST_GET_PAGE\n");

            if (copy_from_user((void *)&(cPageParams), (void *)arg, sizeof(cPageParams)) != 0)
            {
                // copy failed
                ret = -EFAULT;
            }
            else
            {
                ret = hbm_fpga_pagetable_test_get_page(fpga, &cPageParams);
            }
            break;

        case HBM_FPGA_IOCTL_TEST_PUT_PAGE:
//          dev_dbg(&fpga->pci_dev->dev, "hbm_fpga_chrdev_ioctl: HBM_FPGA_IOCTL_TEST_PUT_PAGE\n");

            if (copy_from_user((void *)&(cPageParams), (void *)arg, sizeof(cPageParams)) != 0)
            {
                // copy failed
                ret = -EFAULT;
            }
            else
            {
                ret = hbm_fpga_pagetable_test_put_page(fpga, &cPageParams);
            }
            break;

        case HBM_FPGA_IOCTL_UNLOCK_COMPOUND_PAGE:
//          dev_dbg(&fpga->pci_dev->dev, "hbm_fpga_chrdev_ioctl: HBM_FPGA_IOCTL_UNLOCK_COMPOUND_PAGE\n");

            if (copy_from_user((void *)&(cPageParams), (void *)arg, sizeof(cPageParams)) != 0)
            {
                // copy failed
                ret = -EFAULT;
            }
            else
            {
                ret = hbm_fpga_pagetable_unlock_compound_page(fpga, &cPageParams);
            }
            break;

        default:
            // return: "inappropriate ioctl for device"
            ret = -ENOTTY;
            break;
    }

    mutex_unlock(&(fpga->driver_mutex)) ;
    return ret;
}

static int hbm_fpga_mem_fault(struct vm_area_struct *vma, struct vm_fault *vmf)
{
    struct hbm_fpga_dev * fpga = NULL;
    int page_nr = 0;
    unsigned long phys_offset;
    unsigned long phys_addr;

    fpga = (struct hbm_fpga_dev *) vma->vm_private_data;

    BUG_ON(!fpga);

#ifdef ZBUF_CPUSHARED
    if (vma == fpga->vmaCpuShared)
        return hbm_cpushared_mem_fault (vma, vmf);
#endif /* ZBUF_CPUSHARED */

    dev_dbg(&fpga->pci_dev->dev, "hbm_fpga_mem_fault: virtual address = 0x%08x pg_off = 0x%08x\n",
        (unsigned int)vmf->virtual_address, (unsigned int)vmf->pgoff);

    // Sanity checking
    if ((unsigned long)vmf->virtual_address < vma->vm_start)
    {
        return VM_FAULT_SIGBUS;
    }

    // Calculate offset
    phys_offset = (unsigned long)vmf->virtual_address - vma->vm_start;

    // Offset should be >= 128 MB, because RAM is mapped from here...
    if (phys_offset < FPGA_DATA_BUFFER_OFFSET)
    {
        return VM_FAULT_SIGBUS;
    }

    // Calculate physical address
    phys_addr = fpga->fpga_mem_base_phys + phys_offset;

    // Subtract RAM offset
    phys_offset -= FPGA_DATA_BUFFER_OFFSET;

    // Calculate page number
    page_nr = phys_offset >> PAGE_SHIFT;

    dev_dbg(&fpga->pci_dev->dev, "hbm_fpga_mem_fault: offset = 0x%08x, page_nr = 0x%x\n",
        (unsigned int)phys_offset, page_nr);

    if (page_nr >= EXT_NR_RAM_PAGES)
    {
        return VM_FAULT_SIGBUS;
    }

    // Return page
    vmf->page = &fpga->pExt_ram_page[page_nr];

    return 0;
}

static const struct vm_operations_struct hbm_fpga_vm_ops = {
    .fault = hbm_fpga_mem_fault
};

static int hbm_fpga_chrdev_mmap(struct file *f, struct vm_area_struct *vma)
{
    unsigned long phys_addr;
    unsigned long phys_size;
    struct hbm_fpga_dev * fpga = NULL;
    fpga = (struct hbm_fpga_dev *) f->private_data;

    BUG_ON(!f);
    BUG_ON(!vma);
    BUG_ON(!fpga);

#ifdef ZBUF_CPUSHARED
    if (vma->vm_end - vma->vm_start == CPUSHARED_MEMORY_SIZE)
        return hbm_cpushared_chrdev_mmap(f, vma);
    fpga->vmaFpga = vma;
#endif /* ZBUF_CPUSHARED */

    dev_dbg(&fpga->pci_dev->dev, "hbm_fpga_chrdev_mmap\n");

    phys_addr = fpga->fpga_mem_base_phys;
    phys_size = fpga->fpga_mem_size;

    dev_dbg(&fpga->pci_dev->dev, "hbm_fpga_chrdev_mmap: phys_mem_base: 0x%x, size: 0x%x\n", (unsigned int)phys_addr, (unsigned int)phys_size);

    if (remap_pfn_range(vma, vma->vm_start, phys_addr >> PAGE_SHIFT, phys_size, vma->vm_page_prot))
    {
        dev_err(&fpga->pci_dev->dev, "hbm_fpga_chrdev_mmap: failed\n");
        return -EAGAIN;
    }

    // Store VM operations & private data
    vma->vm_ops = &hbm_fpga_vm_ops;
    vma->vm_private_data = fpga;

    // store virtual base address
    dev_dbg(&fpga->pci_dev->dev, "hbm_fpga_chrdev_mmap: vma->vm_start: 0x%x vma->flags:0x%08x\n",
        (unsigned int)vma->vm_start, (unsigned int)vma->vm_flags);

    return 0;
}

static irqreturn_t hbm_fpga_cmdnow_isr(int irq, void * dev_id)
{
    struct hbm_fpga_dev *fpga = (struct hbm_fpga_dev *) dev_id;
    unsigned int TCDR_value;

    BUG_ON(!fpga);
    BUG_ON(!fpga->pTimebase);

    // Read TCDR value
    TCDR_value = fpga->pTimebase->TCDR;

    if (fpga->chardev[FPGA_CHARDEV_MINOR_CMD_NOW].is_active > 0)
    {
        // Store TCDR
        if (TCDR_SIZE != kfifo_in(fpga->chardev[FPGA_CHARDEV_MINOR_CMD_NOW].fifo_in,
                (unsigned char *)&TCDR_value, TCDR_SIZE))
        {
            dev_dbg(&fpga->pci_dev->dev, "hbm_fpga_cmdnow_isr: kfifo_put failed\n");
        }
        else
        {
//          dev_dbg(&fpga->pci_dev->dev, "hbm_fpga_cmdnow_isr: waking up wait_queue\n");

            wake_up_interruptible(&fpga->chardev[FPGA_CHARDEV_MINOR_CMD_NOW].wait_queue);
        }
    }

    return IRQ_HANDLED;
}

static irqreturn_t hbm_fpga_timebase_isr(int irq, void * dev_id)
{
    struct hbm_fpga_dev *   fpga = (struct hbm_fpga_dev *) dev_id;
    TIMEBASE_STRUCT         timebase_struct;
    unsigned int            int_mask = ((1 << TIMEBASE_1PPS_INT) | (1 << TIMEBASE_TIMEBASE_INT));

    BUG_ON(!fpga);
    BUG_ON(!fpga->pTimebase);

    timebase_struct.TSR = fpga->pTimebase->TSR;
    timebase_struct.TSIR = fpga->pTimebase->TSIR;
    timebase_struct.TS0R = fpga->pTimebase->TS0R;
    timebase_struct.TIDR = fpga->pTimebase->TIDR;

//  dev_dbg(&fpga->pci_dev->dev, "hbm_fpga_timebase_isr: TSR = 0x%08x, TSIR = 0x%08x TS0R = 0x%08x TIDR = 0x%08x\n", timebase_struct.TSR, timebase_struct.TSIR, timebase_struct.TS0R, timebase_struct.TIDR);

    if (fpga->chardev[FPGA_CHARDEV_MINOR_TIMEBASE].is_active > 0)
    {
        if ((timebase_struct.TSR & int_mask) != 0)
        {
            // read + store timebase info in timebase fifo!
            if (sizeof(TIMEBASE_STRUCT) != kfifo_in(fpga->chardev[FPGA_CHARDEV_MINOR_TIMEBASE].fifo_in,
                    (unsigned char *)&timebase_struct, sizeof(TIMEBASE_STRUCT)))
            {
                dev_dbg(&fpga->pci_dev->dev, "hbm_fpga_timebase_isr: kfifo_put failed\n");
            }
            else
            {
//              dev_dbg(&fpga->pci_dev->dev, "hbm_fpga_timebase_isr: waking up wait_queue\n");

                wake_up_interruptible(&fpga->chardev[FPGA_CHARDEV_MINOR_TIMEBASE].wait_queue);
            }
        }
        else
        {
            dev_err(&fpga->pci_dev->dev, "hbm_fpga_timebase_isr: Not my interrupt?!?\n");
        }
    }

    return IRQ_HANDLED;
}

static irqreturn_t hbm_fpga_masterslave_isr(int irq, void * dev_id)
{
    struct hbm_fpga_dev *fpga = (struct hbm_fpga_dev *) dev_id;
    unsigned int MSMSGSTAT_value;
    unsigned int MSINTSTAT_value;

    BUG_ON(!fpga);
    BUG_ON(!fpga->pMasterSlave);

    // First read MSINTSTAT to clear the interrupt...
    MSINTSTAT_value = fpga->pMasterSlave->MSINTSTAT;

    // ... then read the MSMSGSTAT value. This sequence is important, otherwise we could miss level changes.
    MSMSGSTAT_value = fpga->pMasterSlave->MSMSGSTAT;

    if (fpga->chardev[FPGA_CHARDEV_MINOR_MASTERSLAVE].is_active > 0)
    {
        // Store MSMSGSTAT
        if (MSMSGSTAT_SIZE != kfifo_in(fpga->chardev[FPGA_CHARDEV_MINOR_MASTERSLAVE].fifo_in,
                (unsigned char *)&MSMSGSTAT_value, MSMSGSTAT_SIZE))
        {
            dev_dbg(&fpga->pci_dev->dev, "hbm_fpga_masterslave_isr: kfifo_put failed\n");
        }
        else
        {
//          dev_dbg(&fpga->pci_dev->dev, "hbm_fpga_masterslave_isr: waking up wait_queue\n");

            wake_up_interruptible(&fpga->chardev[FPGA_CHARDEV_MINOR_MASTERSLAVE].wait_queue);
        }
    }

    return IRQ_HANDLED;
}

static irqreturn_t hbm_fpga_msgqueue_isr(int irq, void * dev_id)
{
    struct hbm_fpga_dev *fpga = (struct hbm_fpga_dev *) dev_id;

    BUG_ON(!fpga);

    if (fpga->chardev[FPGA_CHARDEV_MINOR_MSG_QUEUE].is_active > 0)
    {
//      dev_dbg(&fpga->pci_dev->dev, "hbm_fpga_msgqueue_isr\n");

        tasklet_schedule(&fpga->chardev[FPGA_CHARDEV_MINOR_MSG_QUEUE].tasklet);
    }

    return IRQ_HANDLED;
}

/* Handler function for the tasklet; the real work is done here. */
static void hbm_fpga_msgqueue_tasklet_handler(unsigned long data)
{
    struct hbm_fpga_dev *   fpga = (struct hbm_fpga_dev*) data;
    unsigned int            frameOffset;

    BUG_ON(!fpga);
    BUG_ON(!fpga->pFPGAMsgQ);
    BUG_ON(!fpga->pSystem);

/*    fpga->pDebug->SCRATCH1 = 0x20;*/

    // No locking required, tasklet doesn't execute concurrently

    // According to kfifo.c, no kfifo locking is required when using only
    // 1 concurrent reader and 1 concurrent writer

//  dev_dbg(&fpga->pci_dev->dev, "MSGQ tasklet started\n");

/*    fpga->pDebug->SCRATCH1 = 0x21;*/

    // Read all frameOffsets from receive FIFO

    // Check if data is available
    frameOffset = fpga->pFPGAMsgQ->recvMsgQ;

/*    fpga->pDebug->SCRATCH1 = 0x22;*/

    while (frameOffset != RECV_FIFO_EMPTY_VALUE)
    {
//      dev_dbg(&fpga->pci_dev->dev, "MSGQ tasklet: frameOffset: 0x%x\n", frameOffset);

        /* write to fifo */
        if (FRAME_OFFSET_SIZE != kfifo_in(fpga->chardev[FPGA_CHARDEV_MINOR_MSG_QUEUE].fifo_in,
                                (unsigned char *)&frameOffset, FRAME_OFFSET_SIZE)) {

            /* Put Jeroen's KRAK here, no space left if FIFO? */
            dev_err(&fpga->pci_dev->dev, "MSGQ tasklet: kfifo_put failed\n");

            // TODO: how to handle FIFO full??
            //BUG();
        }

/*        fpga->pDebug->SCRATCH1 = 0x23;*/

        wake_up_interruptible(&fpga->chardev[FPGA_CHARDEV_MINOR_MSG_QUEUE].wait_queue);

/*        fpga->pDebug->SCRATCH1 = 0x24;*/

        frameOffset = fpga->pFPGAMsgQ->recvMsgQ;

/*        fpga->pDebug->SCRATCH1 = 0x25;*/
    }

/*    fpga->pDebug->SCRATCH1 = 0x26;*/

//  dev_dbg(&fpga->pci_dev->dev, "MSGQ tasklet ended\n");

    // Enable MSGQ interrupt
    fpga->pSystem->INTEN |= (1 << INT_MSGQ);

/*    fpga->pDebug->SCRATCH1 = 0x27;*/
}

/* Reset FPGA MsgQ FIFOs */
static void hbm_fpga_reset_fifos(struct hbm_fpga_dev* fpga)
{
    BUG_ON(!fpga);
    BUG_ON(!fpga->pFPGAMsgQ);

    // Reset FIFO's
    fpga->pFPGAMsgQ->MSGCR = 1 << MSGCR_RESET_FIFO;
}

/* Initialize all structures for the Message Queue character device implementation */
static int hbm_fpga_init_msgqueue_chardev(struct hbm_fpga_dev* fpga)
{
    //volatile unsigned int intMask = 0;
    unsigned int        frameOffset = 0;
    unsigned char  *    pDataBuffer = 0;
    int i = 0;

    BUG_ON(!fpga);
    BUG_ON(!fpga->pFPGAMsgQ);

    /* Init the spinlock for this chardev */
//  spin_lock_init(&fpga->chardev[FPGA_CHARDEV_MINOR_MSG_QUEUE].spinlock_fifo);
//  spin_lock_init(&fpga->chardev[FPGA_CHARDEV_MINOR_MSG_QUEUE].spinlock_read);
//  spin_lock_init(&fpga->chardev[FPGA_CHARDEV_MINOR_MSG_QUEUE].spinlock_write);

    /* Init the kfifo that will contain our read messages */
    fpga->chardev[FPGA_CHARDEV_MINOR_MSG_QUEUE].fifo_in = &fpga->chardev[FPGA_CHARDEV_MINOR_MSG_QUEUE].fifo ;
            kfifo_alloc(
                &fpga->chardev[FPGA_CHARDEV_MINOR_MSG_QUEUE].fifo,
                FRAME_OFFSET_SIZE * MSQ_FIFO_NR_MSG,
                GFP_KERNEL);

    if (IS_ERR(fpga->chardev[FPGA_CHARDEV_MINOR_MSG_QUEUE].fifo_in))
    {
        dev_err(&fpga->pci_dev->dev, "hbm_fpga_init_msgqueue_chardev: kfifo_alloc failed\n");
        return (int) (fpga->chardev[FPGA_CHARDEV_MINOR_MSG_QUEUE].fifo_in);
    }

    /* Init the wait queue so we can sleep while waiting for messages */
    init_waitqueue_head(&(fpga->chardev[FPGA_CHARDEV_MINOR_MSG_QUEUE].wait_queue));

    /* Init the tasklet structure, never fails */
    tasklet_init(&(fpga->chardev[FPGA_CHARDEV_MINOR_MSG_QUEUE].tasklet),
                         hbm_fpga_msgqueue_tasklet_handler, (unsigned long) fpga);

    /* Initialise FPGA FIFO's */
    hbm_fpga_reset_fifos(fpga);

    // initialising of buffer with Shared CPU memory

    pDataBuffer = (unsigned char *)(CPUSHARED_MEMORY_START - fpga->fpga_mem_base_phys);

    frameOffset = 0;

    /* Fill free FIFO with frame offsets */
    for (i = 0; i < MSQ_FIFO_NR_MSG; i++)
    {
        /* Check frameOffset not above FPGA_MU_MEMORY_SIZE */
        //BUG_ON((frameOffset - FPGA_MU_MEMORY_OFFSET) >= FPGA_MU_MEMORY_SIZE);

        fpga->pFPGAMsgQ->recvMsgQ = (unsigned int)(pDataBuffer + frameOffset);

        // Calculate next frameOffset
        frameOffset += MSQ_MESSAGE_SIZE_BYTES;
    }
    /* Enable IRQ generation in FPGA, unmask msgqueue interrupt */
    fpga->pSystem->INTEN |= (1 << INT_MSGQ);

    return 0;
}

static int hbm_fpga_release_msgqueue_chardev(struct hbm_fpga_dev* fpga)
{
    // TODO: mask msgqueue int in FPGA

    /* Kill the tasklet */
    tasklet_kill( &(fpga->chardev[FPGA_CHARDEV_MINOR_MSG_QUEUE].tasklet));

    /* Release the fifo */
    if (fpga->chardev[FPGA_CHARDEV_MINOR_MSG_QUEUE].fifo_in)
    {
        kfifo_free(fpga->chardev[FPGA_CHARDEV_MINOR_MSG_QUEUE].fifo_in);
        fpga->chardev[FPGA_CHARDEV_MINOR_MSG_QUEUE].fifo_in = NULL;
    }

    /* clear Msg Q FIFO's */
    hbm_fpga_reset_fifos(fpga);

    /* tear down the wait queue */

    /* Note: no need to destroy spin_lock, simply don't use it anymore! */

    return 0;
}

/* Initialize all structures for the CMD NOW character device implementation */
static int hbm_fpga_init_cmdnow_chardev(struct hbm_fpga_dev* fpga)
{
    BUG_ON(!fpga);
    BUG_ON(!fpga->pSystem);

    /* Init the spinlock for this chardev */
//        spin_lock_init(&fpga->chardev[FPGA_CHARDEV_MINOR_CMD_NOW].spinlock_fifo);
//        spin_lock_init(&fpga->chardev[FPGA_CHARDEV_MINOR_CMD_NOW].spinlock_read);
//        spin_lock_init(&fpga->chardev[FPGA_CHARDEV_MINOR_CMD_NOW].spinlock_write);

    /* Init the kfifo that will contain TCDR register contents */
    fpga->chardev[FPGA_CHARDEV_MINOR_CMD_NOW].fifo_in = &fpga->chardev[FPGA_CHARDEV_MINOR_CMD_NOW].fifo ;

            kfifo_alloc(
                &fpga->chardev[FPGA_CHARDEV_MINOR_CMD_NOW].fifo,
                CMDNOW_FIFO_SIZE * TCDR_SIZE,
                GFP_KERNEL );

    if (IS_ERR(fpga->chardev[FPGA_CHARDEV_MINOR_CMD_NOW].fifo_in))
    {
        dev_err(&fpga->pci_dev->dev, "hbm_fpga_init_cmdnow_chardev: kfifo_alloc failed\n");
        return (int) (fpga->chardev[FPGA_CHARDEV_MINOR_CMD_NOW].fifo_in);
    }

    /* Init the wait queue so we can sleep while waiting for cmdnow*/
    init_waitqueue_head(&(fpga->chardev[FPGA_CHARDEV_MINOR_CMD_NOW].wait_queue));

    /* Enable IRQ generation in FPGA, unmask cmdnow interrupt */
    fpga->pSystem->INTEN |= (1 << INT_CMDNOW);

    return 0;
}

static int hbm_fpga_release_cmdnow_chardev(struct hbm_fpga_dev* fpga)
{
    /* Disable IRQ generation in FPGA */
    fpga->pSystem->INTEN &= ~(1 << INT_CMDNOW);

    /* Release the fifo */
    if (fpga->chardev[FPGA_CHARDEV_MINOR_CMD_NOW].fifo_in)
    {
        kfifo_free(fpga->chardev[FPGA_CHARDEV_MINOR_CMD_NOW].fifo_in);
        fpga->chardev[FPGA_CHARDEV_MINOR_CMD_NOW].fifo_in = NULL;
    }

    /* tear down the wait queue */
// TODO

    /* Note: no need to destroy spin_lock, simply don't use it anymore! */

    return 0;
}

/* Initialize all structures for the Timebase character device implementation */
static int hbm_fpga_init_timebase_chardev(struct hbm_fpga_dev* fpga)
{
    int                 i;
    TIMEBASE_STRUCT     timebase_struct;

    BUG_ON(!fpga);

    /* Init the spinlock for this chardev */
//        spin_lock_init(&fpga->chardev[FPGA_CHARDEV_MINOR_TIMEBASE].spinlock_fifo);
//        spin_lock_init(&fpga->chardev[FPGA_CHARDEV_MINOR_TIMEBASE].spinlock_read);
//        spin_lock_init(&fpga->chardev[FPGA_CHARDEV_MINOR_TIMEBASE].spinlock_write);

    /* Init the kfifo that will contain TSR, TS0R, TIDR register contents */
    fpga->chardev[FPGA_CHARDEV_MINOR_TIMEBASE].fifo_in = &fpga->chardev[FPGA_CHARDEV_MINOR_TIMEBASE].fifo ;
    kfifo_alloc(
            &fpga->chardev[FPGA_CHARDEV_MINOR_TIMEBASE].fifo,
            TIMEBASE_FIFO_SIZE * sizeof(TIMEBASE_STRUCT),
            GFP_KERNEL );

    if (IS_ERR(fpga->chardev[FPGA_CHARDEV_MINOR_TIMEBASE].fifo_in))
    {
        dev_err(&fpga->pci_dev->dev, "hbm_fpga_init_timebase_chardev: kfifo_alloc failed\n");
        return (int) (fpga->chardev[FPGA_CHARDEV_MINOR_TIMEBASE].fifo_in);
    }

    /* Init the wait queue so we can sleep while waiting for timebase */
    init_waitqueue_head(&(fpga->chardev[FPGA_CHARDEV_MINOR_TIMEBASE].wait_queue));

    // Clear timestamp queue's by reading all entries first

    // Note: timestamp queue size = 512 entries
    for (i = 0; i < TIMESTAMP_HW_QUEUE_SIZE; i++)
    {
        timebase_struct.TSR = fpga->pTimebase->TSR;
        timebase_struct.TSIR = fpga->pTimebase->TSIR;
        timebase_struct.TS0R = fpga->pTimebase->TS0R;
        timebase_struct.TIDR = fpga->pTimebase->TIDR;
    }

    /* Enable IRQ generation in FPGA, unmask TIMEBASE interrupt */
    fpga->pSystem->INTEN |= (1 << INT_TIMEBASE);

    return 0;
}

static int hbm_fpga_release_timebase_chardev(struct hbm_fpga_dev* fpga)
{
    /* Disable IRQ generation in FPGA */
    fpga->pSystem->INTEN &= ~(1 << INT_TIMEBASE);

    /* Release the fifo */
    if (fpga->chardev[FPGA_CHARDEV_MINOR_TIMEBASE].fifo_in)
    {
        kfifo_free(fpga->chardev[FPGA_CHARDEV_MINOR_TIMEBASE].fifo_in);
        fpga->chardev[FPGA_CHARDEV_MINOR_TIMEBASE].fifo_in = NULL;
    }

    /* tear down the wait queue */
    // TODO

    /* Note: no need to destroy spin_lock, simply don't use it anymore! */

    return 0;
}

static int hbm_fpga_init_compound_page_chardev(struct hbm_fpga_dev* fpga)
{
    BUG_ON(!fpga);

    /*
     * Init the kfifo that will contain the compound page pointers that need to be
     * freed.
     */
    fpga->chardev[FPGA_CHARDEV_MINOR_COMPOUND_PAGE].fifo_in = &fpga->chardev[FPGA_CHARDEV_MINOR_COMPOUND_PAGE].fifo;
        kfifo_alloc(
                &fpga->chardev[FPGA_CHARDEV_MINOR_COMPOUND_PAGE].fifo,
                MEM_PHYS_ADDR_SIZE * MEM_FREE_FIFO_SIZE,
                GFP_KERNEL );



    if (IS_ERR(fpga->chardev[FPGA_CHARDEV_MINOR_COMPOUND_PAGE].fifo_in))
    {
        dev_err(&fpga->pci_dev->dev, "hbm_fpga_init_compound_page_chardev: kfifo_alloc failed\n");
        return (int) (fpga->chardev[FPGA_CHARDEV_MINOR_COMPOUND_PAGE].fifo_in);
    }

    /* Init the wait queue so we can sleep while waiting for compound pages */
    init_waitqueue_head(&(fpga->chardev[FPGA_CHARDEV_MINOR_COMPOUND_PAGE].wait_queue));

    return 0;
}

static int hbm_fpga_release_compound_page_chardev(struct hbm_fpga_dev* fpga)
{
    /* Re-initialize pagetable to release any left over pages
     * (for debugging mainly).
     */
    hbm_fpga_pagetable_init(fpga);

    /* Release the fifo */
    if (fpga->chardev[FPGA_CHARDEV_MINOR_COMPOUND_PAGE].fifo_in)
    {
        kfifo_free(fpga->chardev[FPGA_CHARDEV_MINOR_COMPOUND_PAGE].fifo_in);
        fpga->chardev[FPGA_CHARDEV_MINOR_COMPOUND_PAGE].fifo_in = NULL;
    }

    /* tear down the wait queue */

    /* Note: no need to destroy spin_lock, simply don't use it anymore! */

    return 0;
}

/* Initialize all structures for the MASTERSLAVE character device implementation */
static int hbm_fpga_init_masterslave_chardev(struct hbm_fpga_dev* fpga)
{
    BUG_ON(!fpga);
    BUG_ON(!fpga->pSystem);

    /* Init the spinlock for this chardev */
//        spin_lock_init(&fpga->chardev[FPGA_CHARDEV_MINOR_MASTERSLAVE].spinlock_fifo);
//        spin_lock_init(&fpga->chardev[FPGA_CHARDEV_MINOR_MASTERSLAVE].spinlock_read);
//        spin_lock_init(&fpga->chardev[FPGA_CHARDEV_MINOR_MASTERSLAVE].spinlock_write);

    /* Init the kfifo that will contain TCDR register contents */
    fpga->chardev[FPGA_CHARDEV_MINOR_MASTERSLAVE].fifo_in = &fpga->chardev[FPGA_CHARDEV_MINOR_MASTERSLAVE].fifo ;
    kfifo_alloc(
            &fpga->chardev[FPGA_CHARDEV_MINOR_MASTERSLAVE].fifo,
            MASTERSLAVE_FIFO_SIZE * MSMSGSTAT_SIZE,
            GFP_KERNEL ) ;

    if (IS_ERR(fpga->chardev[FPGA_CHARDEV_MINOR_MASTERSLAVE].fifo_in))
    {
        dev_err(&fpga->pci_dev->dev, "hbm_fpga_init_masterslave_chardev: kfifo_alloc failed\n");
        return (int) (fpga->chardev[FPGA_CHARDEV_MINOR_MASTERSLAVE].fifo_in);
    }

    /* Init the wait queue so we can sleep while waiting for masterslave */
    init_waitqueue_head(&(fpga->chardev[FPGA_CHARDEV_MINOR_MASTERSLAVE].wait_queue));

    /* Enable IRQ generation in FPGA, unmask masterslave interrupt */
    fpga->pSystem->INTEN |= (1 << INT_MS);

    return 0;
}

static int hbm_fpga_release_masterslave_chardev(struct hbm_fpga_dev* fpga)
{
    /* Disable IRQ generation in FPGA */
    fpga->pSystem->INTEN &= ~(1 << INT_MS);

    /* Release the fifo */
    if (fpga->chardev[FPGA_CHARDEV_MINOR_MASTERSLAVE].fifo_in)
    {
        kfifo_free(fpga->chardev[FPGA_CHARDEV_MINOR_MASTERSLAVE].fifo_in);
        fpga->chardev[FPGA_CHARDEV_MINOR_MASTERSLAVE].fifo_in = NULL;
    }

    /* tear down the wait queue */
    // TODO

    /* Note: no need to destroy spin_lock, simply don't use it anymore! */

    return 0;
}

static int hbm_fpga_chrdev_open(struct inode *i, struct file *f)
{
    int ret = -EINVAL;
    struct hbm_fpga_dev *fpga = container_of(i->i_cdev, struct hbm_fpga_dev, cdev);

    BUG_ON(!fpga);

    // Only Blocking mode supported
    if ((f->f_flags & O_NONBLOCK) == 0)
    {
        // Blocking mode
        SWITCH_MINOR (i)
        {
            case FPGA_CHARDEV_MINOR_MSG_QUEUE:

                // Only one instance allowed
                if (fpga->chardev[FPGA_CHARDEV_MINOR_MSG_QUEUE].is_active == 0)
                {
//                  dev_dbg(&fpga->pci_dev->dev, "hbm_fpga_chrdev_open: FPGA_CHARDEV_MINOR_MSG_QUEUE\n");
                    ret = hbm_fpga_init_msgqueue_chardev(fpga);

                    // Successfull?
                    if (ret == 0)
                    {
                        // set active
                        fpga->chardev[FPGA_CHARDEV_MINOR_MSG_QUEUE].is_active = 1;
                    }
                }
                else
                {
                    dev_err(&fpga->pci_dev->dev, "hbm_fpga_chrdev_open: FPGA_CHARDEV_MINOR_MSG_QUEUE Driver already opened by other application. Only 1 instance allowed\n");
                }

                break;

            case FPGA_CHARDEV_MINOR_CMD_NOW:

                // lock
//              spin_lock(fpga->chardev[FPGA_CHARDEV_MINOR_CMD_NOW].spinlock_read);

//              dev_dbg(&fpga->pci_dev->dev, "hbm_fpga_chrdev_open: FPGA_CHARDEV_MINOR_CMD_NOW\n");
                if ((f->f_flags & O_ACCMODE) != O_RDONLY)
                {
                    /* We can only read from this minor number */
                    ret = -EACCES;
                }
                else
                {
                    // Only one instance allowed
                    if (fpga->chardev[FPGA_CHARDEV_MINOR_CMD_NOW].is_active == 0)
                    {
//                      dev_dbg(&fpga->pci_dev->dev, "hbm_fpga_chrdev_open: FPGA_CHARDEV_MINOR_CMD_NOW\n");
                        ret = hbm_fpga_init_cmdnow_chardev(fpga);

                        // Successfull?
                        if (ret == 0)
                        {
                            // set active
                            fpga->chardev[FPGA_CHARDEV_MINOR_CMD_NOW].is_active = 1;
                        }
                    }
                    else
                    {
                        dev_err(&fpga->pci_dev->dev, "hbm_fpga_chrdev_open: FPGA_CHARDEV_MINOR_CMD_NOW: Driver already opened by other application. Only 1 instance allowed\n");
                    }
                }

//TODO: unlock!!
                break;

            case FPGA_CHARDEV_MINOR_TIMEBASE:

//              dev_dbg(&fpga->pci_dev->dev, "hbm_fpga_chrdev_open: FPGA_CHARDEV_MINOR_TIMEBASE\n");

                if ((f->f_flags & O_ACCMODE) != O_RDONLY)
                {
                    /* We can only read from this minor number */
                    ret =  -EACCES;
                }
                else
                {
                    // Only one instance allowed
                    if (fpga->chardev[FPGA_CHARDEV_MINOR_TIMEBASE].is_active == 0)
                    {
//                      dev_dbg(&fpga->pci_dev->dev, "hbm_fpga_chrdev_open: FPGA_CHARDEV_MINOR_TIMEBASE\n");
                        ret = hbm_fpga_init_timebase_chardev(fpga);

                        // Successfull?
                        if (ret == 0)
                        {
                            // set active
                            fpga->chardev[FPGA_CHARDEV_MINOR_TIMEBASE].is_active = 1;
                        }
                    }
                    else
                    {
                        dev_err(&fpga->pci_dev->dev, "hbm_fpga_chrdev_open: FPGA_CHARDEV_MINOR_TIMEBASE Driver already opened by other application. Only 1 instance allowed\n");
                    }
                }

                break;

            case FPGA_CHARDEV_MINOR_COMPOUND_PAGE:

                // lock
//              spin_lock(fpga->chardev[FPGA_CHARDEV_MINOR_COMPOUND_PAGE].spinlock_read);

//              dev_dbg(&fpga->pci_dev->dev, "hbm_fpga_chrdev_open: FPGA_CHARDEV_MINOR_COMPOUND_PAGE\n");
                if ((f->f_flags & O_ACCMODE) != O_RDONLY)
                {
                    /* We can only read from this minor number */
                    ret = -EACCES;
                }
                else
                {
                    // Only one instance allowed
                    if (fpga->chardev[FPGA_CHARDEV_MINOR_COMPOUND_PAGE].is_active == 0)
                    {
//                      dev_dbg(&fpga->pci_dev->dev, "hbm_fpga_chrdev_open: FPGA_CHARDEV_MINOR_COMPOUND_PAGE\n");
                        ret = hbm_fpga_init_compound_page_chardev(fpga);

                        // Successfull?
                        if (ret == 0)
                        {
                            // set active
                            fpga->chardev[FPGA_CHARDEV_MINOR_COMPOUND_PAGE].is_active = 1;
                        }
                    }
                    else
                    {
                        dev_err(&fpga->pci_dev->dev, "hbm_fpga_chrdev_open: FPGA_CHARDEV_MINOR_COMPOUND_PAGE: Driver already opened by other application. Only 1 instance allowed\n");
                    }
                }

//TODO: unlock!!
                break;

            case FPGA_CHARDEV_MINOR_MASTERSLAVE:

                // lock
//              spin_lock(fpga->chardev[FPGA_CHARDEV_MINOR_MASTERSLAVE].spinlock_read);

//              dev_dbg(&fpga->pci_dev->dev, "hbm_fpga_chrdev_open: FPGA_CHARDEV_MINOR_MASTERSLAVE\n");
                if ((f->f_flags & O_ACCMODE) != O_RDONLY)
                {
                    /* We can only read from this minor number */
                    ret = -EACCES;
                }
                else
                {
                    // Only one instance allowed
                    if (fpga->chardev[FPGA_CHARDEV_MINOR_MASTERSLAVE].is_active == 0)
                    {
//                      dev_dbg(&fpga->pci_dev->dev, "hbm_fpga_chrdev_open: FPGA_CHARDEV_MINOR_MASTERSLAVE\n");
                        ret = hbm_fpga_init_masterslave_chardev(fpga);

                        // Successfull?
                        if (ret == 0)
                        {
                            // set active
                            fpga->chardev[FPGA_CHARDEV_MINOR_MASTERSLAVE].is_active = 1;
                        }
                    }
                    else
                    {
                        dev_err(&fpga->pci_dev->dev, "hbm_fpga_chrdev_open: FPGA_CHARDEV_MINOR_MASTERSLAVE: Driver already opened by other application. Only 1 instance allowed\n");
                    }
                }

//TODO: unlock!!
                break;

            default:
                dev_err(&fpga->pci_dev->dev, "hbm_fpga_chrdev_open: Invalid minor!\n");
                /* Invalid minor, reject open() */
                ret = -EINVAL;
                break;
        }

        if (!ret)
        {
            // Setup private data used when read/write/.. is called
            f->private_data = fpga;
        }
    }
    else
    {
        dev_err(&fpga->pci_dev->dev, "hbm_fpga_chrdev_open: non-blocking mode not supported!\n");

        // non-blocking mode not supported
        ret = -EINVAL;
    }

//  dev_dbg(&fpga->pci_dev->dev, "hbm_fpga_chrdev_open: end\n");
    return ret;
}

static int hbm_fpga_chrdev_release(struct inode *i, struct file *f)
{
    int ret = -EINVAL;
    struct hbm_fpga_dev *fpga = container_of(i->i_cdev, struct hbm_fpga_dev, cdev);
    BUG_ON(!fpga);

    SWITCH_MINOR (i)
    {
        case FPGA_CHARDEV_MINOR_MSG_QUEUE:
//          dev_dbg(&fpga->pci_dev->dev, "hbm_fpga_chrdev_release: FPGA_CHARDEV_MINOR_MSG_QUEUE\n");

            // set inactive
            fpga->chardev[FPGA_CHARDEV_MINOR_MSG_QUEUE].is_active = 0;

            ret = hbm_fpga_release_msgqueue_chardev(fpga);

            break;

        case FPGA_CHARDEV_MINOR_CMD_NOW:
//          dev_dbg(&fpga->pci_dev->dev, "hbm_fpga_chrdev_release: FPGA_CHARDEV_MINOR_CMD_NOW\n");

            // set inactive
            fpga->chardev[FPGA_CHARDEV_MINOR_CMD_NOW].is_active = 0;

            ret = hbm_fpga_release_cmdnow_chardev(fpga);

            break;

        case FPGA_CHARDEV_MINOR_TIMEBASE:
//          dev_dbg(&fpga->pci_dev->dev, "hbm_fpga_chrdev_release: FPGA_CHARDEV_MINOR_TIMEBASE\n");

            // set inactive
            fpga->chardev[FPGA_CHARDEV_MINOR_TIMEBASE].is_active = 0;

            ret = hbm_fpga_release_timebase_chardev(fpga);

            break;

        case FPGA_CHARDEV_MINOR_COMPOUND_PAGE:
//          dev_dbg(&fpga->pci_dev->dev, "hbm_fpga_chrdev_release: FPGA_CHARDEV_MINOR_COMPOUND_PAGE\n");

            // set inactive
            fpga->chardev[FPGA_CHARDEV_MINOR_COMPOUND_PAGE].is_active = 0;

            ret = hbm_fpga_release_compound_page_chardev(fpga);

            break;

        case FPGA_CHARDEV_MINOR_MASTERSLAVE:
//          dev_dbg(&fpga->pci_dev->dev, "hbm_fpga_chrdev_release: FPGA_CHARDEV_MINOR_MASTERSLAVE\n");

            // set inactive
            fpga->chardev[FPGA_CHARDEV_MINOR_MASTERSLAVE].is_active = 0;

            ret = hbm_fpga_release_masterslave_chardev(fpga);

            break;

        default:
            dev_err(&fpga->pci_dev->dev, "hbm_fpga_chrdev_release: Invalid minor!\n");

            /* Invalid minor, reject release() */
            ret = -EINVAL;
            break;
    }

    return ret;
}

struct file_operations hbm_fpga_chrdev_fops = {
    .read       = hbm_fpga_chrdev_read,
    .write      = hbm_fpga_chrdev_write,
    .poll       = NULL,             // Not supported
    .unlocked_ioctl     = hbm_fpga_chrdev_ioctl,
    .mmap       = hbm_fpga_chrdev_mmap,
    .open       = hbm_fpga_chrdev_open,
    .release    = hbm_fpga_chrdev_release,
    .owner      = THIS_MODULE,
};

static int setup_character_devices(struct hbm_fpga_dev *fpga)
{
    int ret;

    BUG_ON(fpga == NULL);
    BUG_ON(fpga->pci_dev == NULL);

    ret = alloc_chrdev_region(&fpga->chrdev_region, 0, FPGA_CHARDEV_MINOR_SIZE, MODNAME);
    if (ret)
    {
        /* Could not allocate major/minor region for this device */
        dev_err( &fpga->pci_dev->dev, "Could not allocate major/minor region for this device");
        return ret;
    }

    cdev_init(&fpga->cdev, &hbm_fpga_chrdev_fops);
    fpga->cdev.owner = THIS_MODULE;

    ret = cdev_add(&fpga->cdev, fpga->chrdev_region, FPGA_CHARDEV_MINOR_SIZE);
    if (ret)
    {
        kobject_put(&fpga->cdev.kobj);
        unregister_chrdev_region(fpga->chrdev_region, FPGA_CHARDEV_MINOR_SIZE);
        dev_err( &fpga->pci_dev->dev, "cdev_add failed");
        return ret;
    }

    // initialize active counters
    fpga->chardev[FPGA_CHARDEV_MINOR_MSG_QUEUE].is_active = 0;
    fpga->chardev[FPGA_CHARDEV_MINOR_CMD_NOW].is_active = 0;
    fpga->chardev[FPGA_CHARDEV_MINOR_TIMEBASE].is_active = 0;
    fpga->chardev[FPGA_CHARDEV_MINOR_COMPOUND_PAGE].is_active = 0;
    fpga->chardev[FPGA_CHARDEV_MINOR_MASTERSLAVE].is_active = 0;

    // reset FIFO
    hbm_fpga_reset_fifos(fpga);

    return 0;
}


static int remove_character_devices(struct hbm_fpga_dev *fpga)
{
    BUG_ON(!fpga);

//  dev_dbg(&fpga->pci_dev->dev, "remove_character_devices\n");

    cdev_del(&fpga->cdev);
    unregister_chrdev_region(fpga->chrdev_region, FPGA_CHARDEV_MINOR_SIZE);

    return 0;
}


/*
    PCI DRIVER RELATED SECTION
*/


static struct pci_device_id hbm_fpga_pci_tbl[] =
{
    // IM2
    {   .vendor = PCI_VENDOR_ID_XILINX,     // from: <include/linux/pci_ids.h> !
        .device = FPGA_PCI_DEVICE_ID,
        .subvendor = FPGA_PCI_SUBVENDOR_ID,
        .subdevice = FPGA_PCI_SUBDEVICE_ID,
    },
    {0,}
};
MODULE_DEVICE_TABLE (pci, hbm_fpga_pci_tbl);

/**
 * THE pci device interrupt service routine handler.
 * Make sure that this function returns as fast as possible! Please use a
 * work queue or a tasklet to execute more time consuming work...
*/
static irqreturn_t hbm_fpga_irq_handler(int irq, void *dev_id)
{
    irqreturn_t             ret = IRQ_NONE;
    unsigned int            interrupt_source = 0;
    unsigned int            interrupt_mask = 0;
    unsigned int            interrupt_status = 0;
    struct hbm_fpga_dev *   fpga = (struct hbm_fpga_dev *) dev_id;

    BUG_ON(!fpga);
    BUG_ON(!fpga->pSystem);
    BUG_ON(!fpga->pFPGAMsgQ);

/*    fpga->pDebug->SCRATCH1 = 0x01;*/

    interrupt_mask = fpga->pSystem->INTEN;
    interrupt_source = fpga->pSystem->INTSR;
    interrupt_status = interrupt_source & interrupt_mask;

/*    fpga->pDebug->SCRATCH1 = 0x02;*/

    if ((fpga->traceFlags & FPGA_TRACE_IRQ) != 0)
    {
        dev_info(&fpga->pci_dev->dev, "hbm_fpga_irq_handler: source: 0x%x, mask = 0x%x, status = 0x%x\n",
            interrupt_source, interrupt_mask, interrupt_status);
    }

/*    fpga->pDebug->SCRATCH1 = 0x03;*/

    // Any interrupt source active
    if (interrupt_status != 0)
    {
        if ((interrupt_status & (1 << INT_CMDNOW)) != 0)
        {
            // CMDNOW interrupt

            // Disable Interrupt
            fpga->pSystem->INTEN &= ~(1 << INT_CMDNOW);

/*            fpga->pDebug->SCRATCH1 = 0x04;*/

            // Handle interrupt
            ret = hbm_fpga_cmdnow_isr(irq, dev_id);

            // Enable Interrupt
            fpga->pSystem->INTEN |= (1 << INT_CMDNOW);
        }

        if ((interrupt_status & (1 << INT_TIMEBASE)) != 0)
        {
            // Timebase interrupt source is one of the following two:
            // - 1PPS
            // - Timestamp on Command event (Timetag)
            // - Top Dead Center event (Timetag)
            // - Command Now event (Timetag)
            // - External clock 0 counter rollover
            // - Timestamp counter rollover

            // Disable Interrupt
            fpga->pSystem->INTEN &= ~(1 << INT_TIMEBASE);

/*            fpga->pDebug->SCRATCH1 = 0x05;*/

            // Handle interrupt
            ret = hbm_fpga_timebase_isr(irq, dev_id);

            // Enable Interrupt
            fpga->pSystem->INTEN |= (1 << INT_TIMEBASE);
        }

        if ((interrupt_status & (1 << INT_MSGQ)) != 0)
        {
            // Disable MSGQ interrupt
            fpga->pSystem->INTEN &= ~(1 << INT_MSGQ);

/*            fpga->pDebug->SCRATCH1 = 0x06;*/

//          dev_dbg(&fpga->pci_dev->dev, "hbm_fpga_irq_handler: MSGQ int\n");

            // MSGQ interrupt
            ret = hbm_fpga_msgqueue_isr(irq, dev_id);
        }

        if ((interrupt_status & (1 << INT_MS)) != 0)
        {
            // MASTERSLAVE interrupt

            // Disable Interrupt
            fpga->pSystem->INTEN &= ~(1 << INT_MS);

/*            fpga->pDebug->SCRATCH1 = 0x09;*/

            // Handle interrupt
            ret = hbm_fpga_masterslave_isr(irq, dev_id);

            // Enable Interrupt
            fpga->pSystem->INTEN |= (1 << INT_MS);
        }

        // Process I2C last, because this one is handled by another interrupt
        // handler!
        if ((interrupt_status & (1 << INT_I2C)) != 0)
        {
/*            fpga->pDebug->SCRATCH1 = 0x07;*/

//          dev_dbg(&fpga->pci_dev->dev, "hbm_fpga_irq_handler: I2C irq\n");

            // Override the status to allow the I2C handled by another ISR
            // handler!
            ret = IRQ_NONE;
        }
    }
    else
    {
/*        fpga->pDebug->SCRATCH1 = 0x08;*/

//      dev_dbg(&fpga->pci_dev->dev, "hbm_fpga_irq_handler: not our interrupt\n");

        // Not our interrupt
        ret = IRQ_NONE;
    }

/*    fpga->pDebug->SCRATCH1 = 0x10;*/

/*    fpga->pDebug->SCRATCH1 = 0x11;*/

    return ret;
}

static int hbm_fpga_pci_setup(struct pci_dev *pdev, struct hbm_fpga_dev *fpga)
{
    int ret = EINVAL;
    unsigned long tmp;

    BUG_ON(!pdev);
    BUG_ON(!fpga);

//  dev_dbg(&fpga->pci_dev->dev, "hbm_fpga_pci_setup\n");

    /* Get PCI config info from the Linux kernel (no need to access */
    /* it again via PCI config cycles) */

    fpga->fpga_mem_base_phys = pci_resource_start(pdev, 0);
    tmp = pci_resource_end(pdev, 0);    /* address is of last valid address */

    fpga->fpga_mem_size = tmp - fpga->fpga_mem_base_phys + 1;
    tmp = pci_resource_flags(pdev, 0);

    dev_info(&fpga->pci_dev->dev, "FPGA mem_base_phys: 0x%x, size: 0x%x\n", (unsigned int)fpga->fpga_mem_base_phys, (unsigned int)fpga->fpga_mem_size);

    BUG_ON(0 == (tmp & IORESOURCE_MEM)); /* This chunk must be MEM */

    fpga->fpga_mem_base_virt = ioremap_nocache(fpga->fpga_mem_base_phys, fpga->fpga_mem_size);
    if (! fpga->fpga_mem_base_virt)
    {
        dev_err( &pdev->dev, "Could not ioremap() fpga PCI memory area (phys = 0x%08lx, "
                "size=0x%04x)", fpga->fpga_mem_base_phys, fpga->fpga_mem_size);
        goto pci_setup_error;
    }
    dev_dbg(&pdev->dev, "fpga->fpga_mem_base_virt: 0x%x\n", (unsigned int)fpga->fpga_mem_base_virt);

    /* initialise fpga pointers */
    fpga->pFPGAMsgQ = fpga->fpga_mem_base_virt + FPGA_MSGQ_REGISTER_OFFSET;
    fpga->pSystem = fpga->fpga_mem_base_virt + FPGA_SYSTEM_OFFSET;
    fpga->pTimebase = fpga->fpga_mem_base_virt + FPGA_TIMEBASE_OFFSET;
    fpga->pDebug = fpga->fpga_mem_base_virt + FPGA_DEVELOPMENT_DEBUG_OFFSET;
    fpga->pMasterSlave = fpga->fpga_mem_base_virt + FPGA_MASTER_SLAVE_OFFSET;

    /* Disable IRQ generation in FPGA, mask all interrupts */
    fpga->pSystem->INTEN = 0x0;

    /* Clear any pending IRQ's */
    tmp = fpga->pSystem->INTSR;

    // Clear trace flags
    fpga->traceFlags = FPGA_TRACE_NONE;

    /* Setup the IRQ */
    /* PATCH GR */
    ret = request_irq( pdev->irq, hbm_fpga_irq_handler, IRQF_SHARED, MODNAME, fpga );
    if (ret)
    {
        dev_err( &pdev->dev, "Request of interrupt failed (irq =  0x%02x, ret: %d)", pdev->irq, ret);
        goto pci_setup_error;
    }

    fpga->irq = pdev->irq; /* The kernel already knows the irq number assigned */

    return 0;

pci_setup_error:
    if (fpga->fpga_mem_base_virt)
        iounmap(fpga->fpga_mem_base_virt);
    if (fpga->irq)
        free_irq(fpga->irq, fpga);

    return ret;
}

static int __devinit hbm_fpga_pci_probe(struct pci_dev *pdev,
                       const struct pci_device_id *ent)
{
    struct hbm_fpga_dev *fpga = NULL;
    int ret;

    dev_info( &pdev->dev, "Found HBM FPGA");

    /* Now allocate (zeroed) memory for our internal bookkeeping */
    fpga = kzalloc( sizeof(struct hbm_fpga_dev), GFP_KERNEL);
    if (!fpga)
    {
        dev_err( &pdev->dev, "Could not allocate struct hbm_fpga_dev, aborting...");
        ret = -ENOMEM;
        goto probe_out;
    }

    dev_dbg( &pdev->dev, "Allocating %d number of external RAM pages (of %d bytes each).",
        EXT_NR_RAM_PAGES, (unsigned int)PAGE_SIZE);

    if (EXT_NR_RAM_PAGES * PAGE_SIZE != FPGA_DATA_BUFFER_SIZE)
    {
        dev_err( &pdev->dev, "Number of external RAM pages does not cover the entire external RAM region!");
        goto probe_out_free;
    }

    /* Now allocate (zero-ed) memory for our own memory pages */
#ifdef ZBUF_CPUSHARED
    fpga->pExt_ram_page = kzalloc(sizeof(struct page) * EXT_NR_COMPOUND_PAGES + CPUSHARED_NR_COMPOUND_PAGES, GFP_KERNEL);   /*TODO: Shouldn't this be NR_RAM_PAGES instead of NR_COMPOUND_PAGES ???? */
#else /* ZBUF_CPUSHARED */
    fpga->pExt_ram_page = kzalloc(sizeof(struct page) * EXT_NR_COMPOUND_PAGES, GFP_KERNEL);
#endif /* ZBUF_CPUSHARED */
    if (!fpga->pExt_ram_page)
    {
        dev_err( &pdev->dev, "Could not allocate external memory pages, aborting...");
        ret = -ENOMEM;
        goto probe_out_free;
    }

    pci_set_drvdata (pdev, fpga);
    pci_enable_device(pdev); // TODO: check return value

    ret = pci_request_regions(pdev, MODNAME);
    if (ret)
    {
        dev_err( &pdev->dev, "Could not claim pci device related regions, aborting... (err = %d / 0x%02x)", ret, ret);
        goto probe_out_disable;
    }

    fpga->pci_dev = pdev;
    ret = hbm_fpga_pci_setup(pdev, fpga);
    if (ret)
    {
        dev_err( &pdev->dev, "hbm_fpga_pci_setup() failed, aborting... (err = %d / 0x%02x)", ret, ret);
        goto probe_out_release;
    }

    pci_set_master(pdev);

    ret = setup_character_devices(fpga);
    if (ret)
    {
        dev_err( &pdev->dev, "Could not setup character devices, aborting... (err = %d / 0x%02x)", ret, ret);
        goto probe_out_free;
    }

    ret = hbm_fpga_init_i2c(fpga);
    if (ret)
    {
        dev_err( &pdev->dev, "Could not init hbm-fpga-i2c platform device... (err = %d / 0x%02x)", ret, ret);
        goto probe_out_chardevs;
    }

    // Initialize specific fpga dev pointer
    s_fpga_dev = fpga;

    /* Initialize page table */
    hbm_fpga_pagetable_init(fpga);

    mutex_init( &(fpga->driver_mutex) );

    dev_dbg( &pdev->dev, "hbm_fpga_pci_probe() finished with success.");
    return 0;

probe_out_chardevs:
    remove_character_devices(fpga);
probe_out_release:
    pci_release_regions(pdev);
probe_out_disable:
    pci_disable_device(pdev);
probe_out_free:
    if  (fpga)
    {
        s_fpga_dev = NULL;

        if (fpga->pExt_ram_page)
        {
            kfree(fpga->pExt_ram_page);
        }

        pci_set_drvdata(pdev, NULL);
        kfree(fpga);
    }
probe_out:
    dev_err( &pdev->dev, "hbm_fpga_pci_probe() failed... (err = %d / 0x%02x)", ret, ret);

    return ret;
}


static void __devexit hbm_fpga_pci_remove(struct pci_dev *pdev)
{
    struct hbm_fpga_dev * fpga = NULL;

    BUG_ON(!pdev);

    fpga = pci_get_drvdata(pdev);

    BUG_ON(!fpga);

    // First disable irq
    if (fpga->irq)
        free_irq(fpga->irq, fpga);

    dev_dbg(&pdev->dev, "hbm_fpga_pci_remove\n");
    pci_set_drvdata(pdev, NULL);

    remove_character_devices(fpga);
    hbm_fpga_uninit_i2c(fpga);

    pci_clear_master(pdev);
    pci_release_regions(pdev);
    pci_disable_device(pdev);

    if(fpga->fpga_mem_base_virt)
        iounmap(fpga->fpga_mem_base_virt);

    s_fpga_dev = NULL;

    kfree(fpga->pExt_ram_page);
    kfree(fpga);
}


#ifdef CONFIG_PM
/*  The following functions are for power management ( PM ) */

static int hbm_fpga_pci_suspend(struct pci_dev *pdev, pm_message_t state)
{
    struct hbm_fpga_dev *dev =  pci_get_drvdata(pdev);

    // TODO: power down / suspend the FPGA hardware

    // TODO: found out what these next 2 lines do
    pci_save_state(pdev);
    pci_set_power_state(pdev, PCI_D3hot);

    return 0;   /* Never return anything else since it will prevent */
                /* the kernel from going into suspend mode! */
}

static int hbm_fpga_pci_resume(struct pci_dev *pdev)
{
    struct hbm_fpga_dev *dev =  pci_get_drvdata(pdev);

    pci_set_power_state(pdev, PCI_D0);
    pci_restore_state(pdev);
    return 0;
}
#endif /* CONFIG_PM */

static struct pci_driver hbm_fpga_pci_driver = {
    .name       = MODNAME,
    .id_table   = hbm_fpga_pci_tbl,
    .probe      = hbm_fpga_pci_probe,
    .remove     = __devexit_p(hbm_fpga_pci_remove),
#ifdef CONFIG_PM
    .suspend    = hbm_fpga_pci_suspend,
    .resume     = hbm_fpga_pci_resume,
#endif /* CONFIG_PM */
};

static int __init hbm_fpga_core_init_module(void)
{
    printk("HBM FPGA core Driver: version: %s, Build date: %s %s\n", DRIVER_VERSION, __TIME__, __DATE__);

    /* when a module, this is printed whether or not devices are found in probe */
    return pci_register_driver(&hbm_fpga_pci_driver);
}

static void __exit hbm_fpga_core_cleanup_module(void)
{
    pci_unregister_driver(&hbm_fpga_pci_driver);
}

MODULE_AUTHOR("Nathan Huizinga <nathan.huizinga@chess.nl>");
MODULE_DESCRIPTION("HBM IntfMod2 FPGA core driver");
MODULE_LICENSE("GPL");

module_init(hbm_fpga_core_init_module);
module_exit(hbm_fpga_core_cleanup_module);


#ifdef ZBUF_CPUSHARED


static void hbm_cpushared_pagetable_init(struct hbm_fpga_dev *fpga)
{
    unsigned long phys_addr = CPUSHARED_MEMORY_START;
    struct page * pPage;
    int i;

    dev_dbg(&fpga->pci_dev->dev, "hbm_cpushared_pagetable_init");

    /*
     * Initialize page table spinlock.
     */
    /* spin_lock_init(&fpga->spinlock_pagetable); already initialized */

    /*
     * Initialize page table:
     * - Mark every page private and owned by us
     * - Pre-calculate physical address of every page
     */
    for (i = EXT_NR_COMPOUND_PAGES; i < EXT_NR_COMPOUND_PAGES + CPUSHARED_NR_COMPOUND_PAGES; i++)
    {
        pPage = &fpga->pExt_ram_page[i];

        // Mark this page as 'special', owned by us (=private ;-).
        SetPageOwnerPriv1(pPage);

        // Set private field containing the physical address of the page
        SetPagePrivate(pPage);
        set_page_private(pPage, phys_addr);

        // Increment to the next physical address
        phys_addr += PAGE_SIZE;
    }
}

static int hbm_cpushared_pagetable_get_page_params(struct hbm_fpga_dev * fpga, compound_page_params * pcPageParams,
    unsigned int * pPageStart, unsigned int * pPageOffset, unsigned int * pNrPages)
{
    int ret = 0;
    unsigned long cpushared_ram_base_phys = CPUSHARED_MEMORY_START;

    if ( (pcPageParams->address < cpushared_ram_base_phys) ||
         (pcPageParams->size == 0) ||
         ((pcPageParams->address + pcPageParams->size) >= (CPUSHARED_MEMORY_START + CPUSHARED_MEMORY_SIZE)) )
    {
        ret = -EINVAL;
    }
    else
    {
        /*
         * Calculate first page and offset within the page of the requested range.
         * - Subtract physical base address from the range start address
         */
        unsigned int page_start = PFN_DOWN(pcPageParams->address - cpushared_ram_base_phys);
        unsigned int page_offset = offset_in_page(pcPageParams->address - cpushared_ram_base_phys);
        unsigned int nr_pages = PFN_UP(pcPageParams->size + page_offset);

        /*
         * Sanity check.
         */
        if ( (page_start + nr_pages) >= CPUSHARED_NR_RAM_PAGES)
        {
            ret = -EINVAL;
        }
        else
        {
            // Return page parameters
            *pPageStart = page_start + EXT_NR_RAM_PAGES;
            *pPageOffset = page_offset;
            *pNrPages = nr_pages;
        }
    }

    return ret;
}

static int hbm_cpushared_mem_fault(struct vm_area_struct *vma, struct vm_fault *vmf)
{
    struct hbm_fpga_dev * fpga = NULL;
    int page_nr = 0;
    unsigned long phys_offset;
    unsigned long phys_addr;

    fpga = (struct hbm_fpga_dev *) vma->vm_private_data;

    BUG_ON(!fpga);

    dev_dbg(&fpga->pci_dev->dev, "hbm_cpushared_mem_fault: virtual address = 0x%08x pg_off = 0x%08x\n",
        (unsigned int)vmf->virtual_address, (unsigned int)vmf->pgoff);

    // Sanity checking
    if ((unsigned long)vmf->virtual_address < vma->vm_start)
    {
        return VM_FAULT_SIGBUS;
    }

    // Calculate offset
    phys_offset = (unsigned long)vmf->virtual_address - vma->vm_start;

    // Calculate physical address
    phys_addr = CPUSHARED_MEMORY_START + phys_offset;

    // Calculate page number
    page_nr = phys_offset >> PAGE_SHIFT;

    page_nr += EXT_NR_RAM_PAGES;

    dev_dbg(&fpga->pci_dev->dev, "hbm_cpushared_mem_fault: offset = 0x%08x, page_nr = 0x%x\n",
        (unsigned int)phys_offset, page_nr);

    if (page_nr >= EXT_NR_RAM_PAGES + CPUSHARED_NR_RAM_PAGES)
    {
        return VM_FAULT_SIGBUS;
    }

    // Return page
    vmf->page = &fpga->pExt_ram_page[page_nr];

    return 0;
}

static int hbm_cpushared_chrdev_mmap(struct file *f, struct vm_area_struct *vma)
{
    unsigned long phys_addr;
    unsigned long phys_size;
    struct hbm_fpga_dev * fpga = NULL;
    fpga = (struct hbm_fpga_dev *) f->private_data;

    BUG_ON(!f);
    BUG_ON(!vma);
    BUG_ON(!fpga);

    dev_dbg(&fpga->pci_dev->dev, "hbm_cpushared_chrdev_mmap\n");

    phys_addr = CPUSHARED_MEMORY_START;
    phys_size = CPUSHARED_MEMORY_SIZE;

    dev_dbg(&fpga->pci_dev->dev, "hbm_cpushared_chrdev_mmap: phys_mem_base: 0x%x, size: 0x%x\n", (unsigned int)phys_addr, (unsigned int)phys_size);

    if (remap_pfn_range(vma, vma->vm_start, phys_addr >> PAGE_SHIFT, phys_size, vma->vm_page_prot))
    {
        dev_err(&fpga->pci_dev->dev, "hbm_cpushared_chrdev_mmap: failed\n");
        return -EAGAIN;
    }

    // Store VM operations & private data
    vma->vm_ops = &hbm_fpga_vm_ops;
    vma->vm_private_data = fpga;
    fpga->vmaCpuShared = vma;

    // store virtual base address
    dev_dbg(&fpga->pci_dev->dev, "hbm_cpushared_chrdev_mmap: vma->vm_start: 0x%x vma->flags:0x%08x\n",
        (unsigned int)vma->vm_start, (unsigned int)vma->vm_flags);

    return 0;
}

#endif /* ZBUF_CPUSHARED */
