/*
 * Copyright (C) 2011 HBM Netherlands B.V.
 * Schutweg 15a
 * 5145NP Waalwijk
 * The Netherlands
 * http://www.hbm.com
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/*
 * HBM FPGA Defines
 */
#ifndef HBM_FPGA_H
#define HBM_FPGA_H

#define FPGA_PCI_DEVICE_ID      (0x1016)    // ?
#define FPGA_PCI_SUBVENDOR_ID   (0x10EE)    // Xilinx
#define FPGA_PCI_SUBDEVICE_ID   (0x3249)    // "I2" Interface Module Mk II

/* Contains the number of messages that should fit in the
   Message Queue fifo and the typical size of each message in bytes.
*/
#define MSQ_MESSAGE_SIZE_BYTES  (1024)
#define MSQ_FIFO_NR_MSG         (128)

/*
 * Timebase/CmdNow FIFO's *MUST* be > 1 to avoid losing an IRQ
 * when starting to process the current one in user space.
 */
#define TIMEBASE_FIFO_SIZE      (10)
#define CMDNOW_FIFO_SIZE        (10)
#define MASTERSLAVE_FIFO_SIZE   (10)

#define MEM_FREE_FIFO_SIZE      (256)
#define MEM_PHYS_ADDR_SIZE      (sizeof(unsigned long))

#define FRAME_OFFSET_SIZE       (sizeof(unsigned int *))
#define TCDR_SIZE               (sizeof(unsigned int))
#define MSMSGSTAT_SIZE          (sizeof(unsigned int))

#define TIMESTAMP_HW_QUEUE_SIZE (512)

#ifdef __cplusplus
extern "C" {
#endif

// TIMEBASE read return struct
typedef struct
{
    unsigned int TSR;
    unsigned int TSIR;
    unsigned int TS0R;
    unsigned int TIDR;
} TIMEBASE_STRUCT;

/** FPGA Register Map
 *
 *  note: all offset are byte offsets
 */
#define FPGA_MSGQ_REGISTER_OFFSET   0x0
typedef struct
{
    unsigned int MSGCR;                 // 0x00
    unsigned int RES0[15];
    unsigned int do_not_use_freeMsgQ;   // 0x40 ACK board side
    unsigned int RES1[3];
    unsigned int recvMsgQ;              // 0x50 IM2 side:
} FPGA_MSGQ_REGISTER_MAP;

#define FPGA_POWER_CONTROL_OFFSET   0x100
typedef struct
{
    unsigned int PCWR_PCSR; // write: PCWR, read: PCSR
} FPGA_POWER_CONTROL_MAP;

#define FPGA_SYSTEM_OFFSET          0x200
typedef struct
{
    unsigned int BSR;           // 0x00
    unsigned int RES0[3];
    unsigned int INTSR;         // 0x10
    unsigned int RES1[3];
    unsigned int INTEN;         // 0x20
    unsigned int RES2[3];
    unsigned int LED;           // 0x30
    unsigned int RES3[3];
    unsigned int IDPROM0;       // 0x40
    unsigned int RES4[3];
    unsigned int IDPROM1;       // 0x50
    unsigned int RES5[3];
    unsigned int I2CPRERL;      // 0x60
    unsigned int I2CPRERH;      // 0x64
    unsigned int I2CCTR;        // 0x68
    unsigned int I2CTXR_RXR;    // 0x6C
    unsigned int I2CCR_SR;      // 0x70
    unsigned int RES6[3];
    unsigned int I2CCTRB;       // 0x80
    unsigned int RES7[3];
    unsigned int FPGAVER;       // 0x90
} FPGA_SYSTEM_MAP;

#define FPGA_TIMEBASE_OFFSET        0x300
typedef struct
{
    unsigned int TCR;       // 0x00
    unsigned int TSR;       // 0x04
    unsigned int RES0[2];
    unsigned int TCDR;      // 0x10
    unsigned int RES1[3];
    unsigned int TRDR;      // 0x20
    unsigned int RES2[3];
    unsigned int TSIR;      // 0x30
    unsigned int TCNT;      // 0x34
    unsigned int RES3[2];
    unsigned int TS0R;      // 0x40
    unsigned int RES4[3];
    unsigned int CGSR;      // 0x50
    unsigned int RES5[3];
    unsigned int TIDR;      // 0x60
    unsigned int RES6[3];
    unsigned int TTMR;      // 0x70
    unsigned int RES7[3];
} FPGA_TIMEBASE_MAP;

#define FPGA_TRIGGER_OFFSET         0x400
typedef struct
{
    unsigned int TRSR;   // TBD
} FPGA_TRIGGER_MAP;

#define FPGA_MASTER_SLAVE_OFFSET    0x500
typedef struct
{
    unsigned int MSCR;      // 0x00
    unsigned int RES0[3];
    unsigned int MSSTAT;    // 0x10
    unsigned int RES1[3];
    unsigned int MSDLY;     // 0x20
    unsigned int RES2[3];
    unsigned int MSINTSTAT; // 0x30
    unsigned int RES3[3];
    unsigned int MSINTEN;   // 0x40
    unsigned int RES4[3];
    unsigned int MSMSGCR;   // 0x50
    unsigned int RES5[3];
    unsigned int MSMSGSTAT; // 0x60
    unsigned int RES6[3];
    unsigned int MSERR;     // 0x70
} FPGA_MASTER_SLAVE_MAP;

#define FPGA_KEYBOARD_OFFSET        0x600
typedef struct
{
    unsigned int KBSCAN_KBSR;   // write: KBSCAN, read: KBSR
} FPGA_KEYBOARD_MAP;

#define FPGA_DEVELOPMENT_DEBUG_OFFSET   0x700
typedef struct
{
    unsigned int SCRATCH1;          // 0x00
    unsigned int RES0[3];
    unsigned int SCRATCH2;          // 0x10
    unsigned int RES1[3];
    unsigned int RXTRIG0_RXDATA0;   // 0x20 write: RXTRIG0, read: RXDATA0
    unsigned int RXTRIG1_RXDATA1;   // 0x24 write: RXTRIG1, read: RXDATA1
    unsigned int RXTRIG2_RXSTAT0;   // 0x28 write: RXTRIG2, read: RXSTAT0
    unsigned int RXARM_RXACK;       // 0x2C write: RXARM, read: RXACK
    unsigned int RXMASK0;           // 0x30
    unsigned int RXMASK1;           // 0x34
    unsigned int RXMASK2;           // 0x38
    unsigned int RXSTOP;            // 0x3C
} FPGA_DEVELOPMENT_DEBUG_MAP;

#define FPGA_SYSTEM_MONITOR_OFFSET  0x900
typedef struct
{
    unsigned int F_TEMP;            // 0x00
    unsigned int F_VCCINT;          // 0x04
    unsigned int F_VCCAUX;          // 0x08
    unsigned int RES0[1];
    unsigned int F_VREFP;           // 0x10
    unsigned int F_VREFN;           // 0x14
    unsigned int RES1[26];
    unsigned int F_TEMPMAX;         // 0x80
    unsigned int F_VCCINTMAX;       // 0x84
    unsigned int F_VCCAUXMAX;       // 0x88
    unsigned int RES2[1];
    unsigned int F_TEMPMIN;         // 0x90
    unsigned int F_VCCINTMIN;       // 0x94
    unsigned int F_VCCAUXMIN;       // 0x98
} FPGA_SYSTEM_MONITOR_MAP;

// Define register bit fields

// MSGCR register bits
#define MSGCR_RESET_FIFO        0

// PCWR
#define PCWR_POWER_DOWN         0

// PCSR
#define PCSR_POWER_DOWN_REQUES  0

// BSR
#define BSR_BOARD_REVISION_MASK (0x03E00000)    // 25:21
#define BSR_SLOT_NUMBER_MASK    (0x001F0000)    // 20:16
#define BSR_BOARD_ID_MASK       (0x0000FFFF)    // 15:0

// INTSR / INTEN
#define INT_TIMEBASE            0
#define INT_CMDNOW              1
#define INT_MSGQ                2
#define INT_MS                  3
#define INT_I2C                 4

// LED
#define LED0                    0
#define LED1                    1
#define LED2                    2
#define LED3                    3

// IDPROM
#define IDPROM_INPUT_DATA_MASK      (0xFF)  // 7:0, read only
#define IDPROM_SELECT               2   // write only
#define IDPROM_SERIAL_OUTPUT_DATA   1   // write only
#define IDPROM_CLOCK                0   // write only

// I2C
//TODO
#define I2C_SDA                     1   // write only
#define I2C_SCL                     0   // write only

// TIMEBASE TCR
#define TIMEBASE_TDC_HOLD_OFF_COUNTER_MASK  (0x0001FE00)    // 16:9, write only
#define TIMEBASE_SAMPLE_ACTIVE              8
#define TIMEBASE_TDC_QUALIFIER              7
#define TIMEBASE_RESYNC_TEST_MODE_DEFAULT   6
#define TIMEBASE_OSC                        5
#define TIMEBASE_RESYNC                     4
#define TIMEBASE_TIME_SOURCE_MASK           (0x0000000C)    // 3:1, write only
#define TIMEBASE_COMMAND_NOW                1

// TIMEBASE TSR
#define TIMEBASE_1PPS_INT                   31  // interrupt cleared after reading TSR!
#define TIMEBASE_TIMEBASE_INT               30  // cleared after reading the timestamp queues!
#define TIMEBASE_1PPS_COUNT_MASK            (0x38000000)    // 29:27, read only
#define TIMEBASE_DIVIDER_VALUE_MASK         (0x07FFFFFF)    // 26:0, read only

// TCDR
#define TCDR_1PPS_COUNT_MASK                (0x38000000)    // 29:27, read only
#define TCDR_DIVIDER_VALUE_MASK             (0x07FFFFFF)    // 26:0, read only

// DACCR
#define DACCR_SELECT                    2
#define DACCR_SERIAL_OUTPUT_DATA        1
#define DACCR_SERIAL_CLOCK              0

// TS0R
#define TS0R_COUNTER_VALUE_MASK         (0x00FFFFFF)    // 23:0

// CGCR
#define CGCR_SELECT                     2
#define CGCR_SERIAL_OUTPUT_DATA         1
#define CGCR_SERIAL_CLOCK               0

// TIDR
//TODO

// TTMR
//TODO

// TRSR
//TODO

// MASTER SLAVE REGISTERS
#define MSCR_BLINK_LEDS                 (1 << 11)
#define MSCR_SW_TRIGGER_ENABLE          (1 << 9)
#define MSCR_RESET_PRBS_ERROR_CNT       (1 << 8)
#define MSCR_CHAN_ENABLE                (1 << 2)
#define MSCR_MASTER_ENABLE              (1 << 1)
#define MSCR_SLAVE_ENABLE               (1 << 0)

#define MSSTAT_PRBS_ERROR               (1 << 1)
#define MSSTAT_LINK_STATUS              (1 << 0)

#define MSDLY_CNTR_HIRES		(1 << 16)
#define MSDLY_CNTR_INVALID		(1 << 15)
#define MSDLY_CNTR_VALUE		((1 << 15) - 1)
#define MSDLY_START			(1 << 0)

#define MSERR_RESET_ERR_CNT             (1 << 0)
#define MSERR_ERR_VALUE                 ( (1 << 4) - 1 )

// KBSCAN
#define KBSCAN_ROW_MASK                 (0x0000000F)    // 3:0, write only

// KBSR
#define KBSR_COLUM_MASK                 (0x0000000F)    // 3:0, read only


// RX Recorder
// TODO

#define FPGA_MEM_SIZE                   0x10000000      // 256M
#define FPGA_DATA_BUFFER_OFFSET         0x08000000      // 128M
#define FPGA_DATA_BUFFER_SIZE           (FPGA_MEM_SIZE - FPGA_DATA_BUFFER_OFFSET) // 128M

#define FPGA_MU_MEMORY_OFFSET           (FPGA_DATA_BUFFER_OFFSET + 0)
#define FPGA_MU_MEMORY_SIZE             (MSQ_FIFO_NR_MSG * MSQ_MESSAGE_SIZE_BYTES)    // fixed space for 128, 128 byte messages (= 16kByte)

#define RECV_FIFO_EMPTY_VALUE           0xFFFFFFFF

//
//      Reserved memory from CPU
//
//#define CPUSHARED_MEMORY_START  0X1E000000
// 56M is maximum shared memory that can be used!
#define CPUSHARED_MEMORY_SIZE   ( 56*1024*1024 )
#define CPUSHARED_MEMORY_START  (0x1F800000-CPUSHARED_MEMORY_SIZE)
#define MSG_FRAME_SIZE           0x00004000






#ifdef __cplusplus
}
#endif

#endif // HBM_FPGA_H
