#!/usr/bin/env bash
#
# Copyright (C) 2011 HBM Netherlands B.V.
# Schutweg 15a
# 5145NP Waalwijk
# The Netherlands
# http://www.hbm.com
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
#

#
# The HBM Interface Module MkII partition table contains the following partitions:
# -1- the boot partition. This is where grub and its config files and the boot kernel reside.
# -2- Software slot 0; the first rootfs image partition. This slot will be filled with the current build rootfs.
# -3- Software slot 1; this slot will be empty and is intended to contain future root filesystems.
# -4- The settings partitions. This partition is to contain an EXT3 file system that is used by software 
#     slot 0 & 1 images to store settings and info persistently.
#

set -x 
[ $# == 7 ] || [ $# == 8 ] || {
    echo "SYNTAX: $0 <file> <kernel size> <kernel directory> <rootfs size> <rootfs image> <settings fs size> <settings dir> [<embed disk image>]"
    exit 1
}

OUTPUT="$1"
KERNELSIZE="$2"
KERNELDIR="$3"
ROOTFSSIZE="$4"
ROOTFSIMAGE="$5"
SETTINGSFSSIZE="$6"
SETTINGSDIR="$7"
EMBEDDISKIMAGE="$8"

rm -f "$OUTPUT"

# Remove any embedded image, present.
rm -f "$KERNELDIR"/openwrt*.gz

TOTALSIZE=$(($KERNELSIZE + 2*($ROOTFSSIZE) + $SETTINGSFSSIZE))
head=16
sect=63
cyl=$(( ($TOTALSIZE) * 1024 * 1024 / ($head * $sect * 512)))

# create partition table
set `ptgen -o "$OUTPUT" -h $head -s $sect -p ${KERNELSIZE}m -p ${ROOTFSSIZE}m -p ${ROOTFSSIZE}m -p ${SETTINGSFSSIZE}m`

KERNELOFFSET="$(($1 / 512))"
KERNELSIZE="$(($2 / 512))"
ROOTFSOFFSET="$(($3 / 512))"
ROOTFSSIZE="$(($4 / 512))"
ROOTFS2OFFSET="$(($5 / 512))"
ROOTFS2SIZE="$(($6 / 512 ))"
SETTINGSFSOFFSET="$(($7 / 512))"
SETTINGSFS2SIZE2="$(($8 / 512 ))"

# Create the disk layout UCI config
install -d ${KERNELDIR}/config
cat > ${KERNELDIR}/config/disklayout <<__SOMEREALLYUNIQUESTRING__
config bootpart boot
	option 'number' 	'1'
	option 'fstype'	'ext2'

config slotpart slot0
	option 'number'		'2'
	option 'slot_number'	'0'
	option 'fstype'	'ext2'

config slotpart slot1
	option 'number'		'3'
	option 'slot_number'	'1'
	option 'fstype'	'ext2'

config settingspart settings
	option 'number'		'4'
	option 'fstype'	'ext2'

__SOMEREALLYUNIQUESTRING__

# Embed the disk image into the kernel image to be able to flash the disk image from USB
[ -n "$EMBEDDISKIMAGE" ] && cp "$EMBEDDISKIMAGE" ${KERNELDIR}

[ -n "$PADDING" ] && dd if=/dev/zero of="$OUTPUT" bs=512 seek="$ROOTFSOFFSET" conv=notrunc count="$ROOTFSSIZE"
# Create the first software slot (filled with freshly built root fs)
dd if="$ROOTFSIMAGE" of="$OUTPUT" bs=512 seek="$ROOTFSOFFSET" conv=notrunc
# Create the second software slot (empty, filled with '\0')
#dd if=/dev/zero of="$OUTPUT" bs=512 count=$ROOTFS2SIZE seek="$ROOTFS2OFFSET" conv=notrunc
# FOR NOW, COPY software slot 0 into slot 1 too!~
[ -n "$PADDING" ] && dd if=/dev/zero of="$OUTPUT" bs=512 seek="$ROOTFS2OFFSET" conv=notrunc count="$ROOTFS2SIZE"
dd if="$ROOTFSIMAGE" of="$OUTPUT" bs=512 seek="$ROOTFS2OFFSET" conv=notrunc

[ -n "$NOGRUB" ] && exit 0

# Create the boot partition
BLOCKS="$((($KERNELSIZE / 2) - 1))"
genext2fs -U -d "$KERNELDIR" -b "$BLOCKS" "$OUTPUT.kernel"
dd if="$OUTPUT.kernel" of="$OUTPUT" bs=512 seek="$KERNELOFFSET" conv=notrunc
rm -f "$OUTPUT.kernel"

# Create the settings fs partition
BLOCKS="$((($SETTINGSFS2SIZE2 / 2) - 1))"

# Create the HBM specific slot config file
rm -rf ${SETTINGSDIR}/boot
mkdir ${SETTINGSDIR}/boot
cat > ${SETTINGSDIR}/boot/slot <<__SOMEREALLYUNIQUESTRING2__
config 'slot_cfg' 'slot0'
	option 'valid' '1'

config 'slot_cfg' 'slot1'
	option 'valid' '1'

config '0' 'active'

__SOMEREALLYUNIQUESTRING2__

# Create md5 file for this config file
md5sum ${SETTINGSDIR}/boot/slot | awk '{printf "%s", $1}' > ${SETTINGSDIR}/boot/slot.md5
echo "  /mnt/settings/boot/slot" >> ${SETTINGSDIR}/boot/slot.md5

# Create the HBM UCI savedir directory
rm -rf ${SETTINGSDIR}/config
mkdir ${SETTINGSDIR}/config
# Enable production tests by default
echo "production.test.enable=1" > ${SETTINGSDIR}/config/production
# Create the HBM production test results directory
mkdir ${SETTINGSDIR}/production-test

genext2fs -U -d "$SETTINGSDIR" -b "$BLOCKS" "$OUTPUT.settings"
dd if="$OUTPUT.settings" of="$OUTPUT" bs=512 seek="$SETTINGSFSOFFSET" conv=notrunc
rm -f "$OUTPUT.settings"

which chpax >/dev/null && chpax -zp $(which grub)
grub --batch --no-curses --no-floppy --device-map=/dev/null <<EOF
device (hd0) $OUTPUT
geometry (hd0) $cyl $head $sect
root (hd0,0)
setup (hd0)
quit
EOF

